/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-geometry project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.geometry.impl;

import java.io.Serializable;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import static terraml.commons.Doubles.isEqual;
import static terraml.commons.Doubles.isSmaller;
import static terraml.commons.Doubles.isSmallerEqual;
import terraml.commons.annotation.File;
import terraml.commons.math.Interval;
import terraml.commons.math.Vec2d;
import terraml.commons.unit.DimensionUnit;
import terraml.geometry.Point2D;
import terraml.geometry.Segment2D;
import terraml.geometry.ShapeUnit;

/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
@File(
        fileName = "ImmutableSegment2D.java",
        packageName = "terraml.geometry.impl",
        projectName = "terraml-geometry"
)
public class ImmutableSegment2D implements Segment2D, Serializable {

    private final Point2D _source;
    private final Point2D _target;

    /**
     * @param double
     * @param double
     * @param double
     * @param double
     */
    public ImmutableSegment2D(double x0, double y0, double x1, double y1) {
        this._source = new ImmutablePoint2D(x0, y0);
        this._target = new ImmutablePoint2D(x1, y1);
    }

    /**
     * @param Point2D
     * @param Point2D
     */
    public ImmutableSegment2D(Point2D _source, Point2D _target) {
        this._source = _source;
        this._target = _target;
    }

    /**
     * @param Segment2D
     */
    public ImmutableSegment2D(Segment2D segment2D) {
        this._source = segment2D.getSource();
        this._target = segment2D.getTarget();
    }

    /**
     * @param Segment2D
     * @param Point2D
     * @param double
     * @return
     */
    protected boolean onRange(Segment2D _s0, Point2D _p0, double _tolerance) {
        final double _srcX = _s0.getSource().getX();
        final double _srcY = _s0.getSource().getY();
        final double _tarX = _s0.getTarget().getX();
        final double _tarY = _s0.getTarget().getY();
        final double _p0x = _p0.getX();
        final double _p0y = _p0.getY();

        if (isEqual(_srcX, _tarX) && isEqual(_srcX, _p0x) && isEqual(_srcY, _tarY) && isEqual(_srcY, _p0y)) {
            return true;
        }

        final double _let0 = (_tarY - _srcY) / (_tarX - _srcX);
        final double _let1 = (_srcY - (_let0 * _srcX));
        final double _let2 = Math.abs(_p0y - (_let0 * _p0x + _let1));

        return isSmaller(_let2, _tolerance);
    }

    @Override
    public boolean onRange(Point2D point2D, double tolerance) {
        return onRange(this, point2D, tolerance);
    }

    /**
     * @param Segment2D
     * @param Point2D
     * @param double
     * @return
     */
    protected boolean inRange(Segment2D _s0, Point2D _p0, double _tolerance) {
        final double _srcX = _s0.getSource().getX();
        final double _srcY = _s0.getTarget().getY();
        final double _p0x = _p0.getX();

        final double _letX = (_srcX <= _srcY) ? _srcX : _srcY;
        final double _letY = (_srcX > _srcY) ? _srcX : _srcY;

        if (onRange(_s0, _p0, _tolerance)) {
            final double cod = (_p0x + _tolerance);

            return isSmallerEqual(cod, _letX) && isSmaller(cod, _letY);
        }

        return false;
    }

    @Override
    public boolean inRange(Point2D point2D, double tolerance) {
        return inRange(this, point2D, tolerance);
    }

    /**
     * @param Segment2D
     * @param Point2D
     * @return
     */
    protected boolean inRange(Segment2D _s0, Point2D _p0) {
        final double _srcX = _s0.getSource().getX();
        final double _srcY = _s0.getTarget().getY();
        final double _tarX = _s0.getTarget().getX();
        final double _tarY = _s0.getTarget().getY();
        final double _p0x = _p0.getX();
        final double _p0y = _p0.getY();

        final double _letX = _tarX - _srcX;
        final double _letY = _tarY - _srcY;
        final double _let = (_p0x - _srcX) * _letX + (_p0y - _srcY) * _letY;

        return isSmallerEqual(0, _let) && isSmallerEqual(_let, (_letX * _letX) + (_letY * _letY));
    }

    @Override
    public boolean inRange(Point2D point2D) {
        return inRange(this, point2D);
    }

    /**
     * @param Segment2D
     * @return
     */
    protected Point2D directionOf(Segment2D _s0) {
        final Vec2d _v0 = _s0.getSource().toVector();
        final Vec2d _v1 = _s0.getTarget().toVector();
        final Vec2d _v2 = _v1.sub(_v0);
        final Vec2d _v3 = _v2.getNormalized();

        return new ImmutablePoint2D(_v3.getX(), _v3.getY());
    }

    @Override
    public Point2D direction() {
        return directionOf(this);
    }

    @Override
    public boolean isVertical() {
        return isEqual(_source.getX(), _target.getX());
    }

    @Override
    public boolean isHorizontal() {
        return isEqual(_source.getY(), _target.getY());
    }

    @Override
    public double horizontalAngle() {
        return _source.angleTo(_target);
    }

    @Override
    public double verticalAngle() {
        final double dx = _target.getX() - _source.getX();
        final double dy = _target.getY() - _source.getY();

        return Math.atan2(dx, dy);
    }

    @Override
    public double getLength() {
        return _source.distanceTo(_target);
    }

    @Override
    public Point2D getMiddle() {
        final double _dx = _source.getX() + _target.getX();
        final double _dy = _source.getY() + _target.getY();

        return new ImmutablePoint2D(_dx / 2, _dy / 2);
    }

    @Override
    public boolean isBounded() {
        return true;
    }

    @Override
    public List<Point2D> getBounds() {
        final Interval _xBound = new Interval(_source.getX(), _target.getX());
        final Interval _yBound = new Interval(_source.getY(), _target.getY());

        Point2D _lowerBounding = new ImmutablePoint2D(_xBound.getLeft(), _yBound.getLeft());
        Point2D _upperBounding = new ImmutablePoint2D(_xBound.getRight(), _yBound.getRight());

        return Arrays.asList(_lowerBounding, _upperBounding);
    }

    @Override
    public DimensionUnit getDimensionUnit() {
        return DimensionUnit.TWO;
    }

    @Override
    public ShapeUnit getShapeUnit() {
        return ShapeUnit.SEGMENT;
    }

    @Override
    public Point2D getSource() {
        return new ImmutablePoint2D(_source);
    }

    @Override
    public Point2D getTarget() {
        return new ImmutablePoint2D(_target);
    }

    @Override
    public Segment2D copy() {
        return new ImmutableSegment2D(getSource(), getTarget());
    }

    @Override
    public Segment2D translate(double... args) {
        Point2D p0 = getSource().translate(args);
        Point2D p1 = getTarget().translate(args);

        return new ImmutableSegment2D(p0, p1);
    }

    @Override
    public Segment2D scale(double scaleFactor) {
        Point2D p0 = getSource().scale(scaleFactor);
        Point2D p1 = getTarget().scale(scaleFactor);

        return new ImmutableSegment2D(p0, p1);
    }

    @Override
    public int hashCode() {
        int hash = 3;
        hash = 67 * hash + Objects.hashCode(this._source);
        hash = 67 * hash + Objects.hashCode(this._target);
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final ImmutableSegment2D other = (ImmutableSegment2D) obj;
        if (!Objects.equals(this._source, other._source)) {
            return false;
        }
        if (!Objects.equals(this._target, other._target)) {
            return false;
        }
        return true;
    }
}
