/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-geometry project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.geometry.impl;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import terraml.commons.Doubles;
import terraml.commons.Ints;
import terraml.commons.Objects;
import terraml.commons.annotation.File;
import terraml.commons.math.Vec2d;
import terraml.commons.unit.DimensionUnit;
import terraml.geometry.Point2D;
import terraml.geometry.Polygon2D;
import terraml.geometry.ShapeUnit;

/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 *
 */
@File(
        fileName = "ImmutablePolygon2D.java",
        packageName = "terraml.geometry.impl",
        projectName = "terraml-geometry"
)
public class ImmutablePolygon2D implements Polygon2D, Serializable {

    private final List<Point2D> _vertices;

    /**
     * @param List<Point2D>
     */
    public ImmutablePolygon2D(List<Point2D> _vertices) {
        this._vertices = _vertices;
    }

    /**
     * @param Polygon2D
     */
    public ImmutablePolygon2D(Polygon2D polygon2D) {
        this._vertices = polygon2D.getVertices();
    }

    /**
     * @param boolean
     * @param boolean
     * @return
     */
    protected int _findMaxOrMin(boolean isMax, boolean isX) {
        double ref = isMax ? 0 : Double.MAX_VALUE;
        int idx = -1;

        for ( int t = 0; t < getVerticesCount(); t++ ) {
            final double temp = isX ? _vertices.get(t).getX() : _vertices.get(t).getY();
            if (isMax) {
                if (temp > ref) {
                    idx = t;
                    ref = temp;
                }
            } else {
                if (temp < ref) {
                    idx = t;
                    ref = temp;
                }
            }
        }

        return idx;
    }

    @Override
    public boolean isConvex() {
        if (Objects.isNull(_vertices) || _vertices.size() <= 4) {
            return false;
        }

        final Polygon2D _instance = (isClosed()) ? open() : this;
        final int _length = _instance.getVerticesCount();

        boolean control = false;
        for ( int t = 0; t < _length; t++ ) {
            // dx1 = vertices.get((t + 2) % _length).getX() - vertices.get((t + 1) % _length).getX();
            // double dy1 = vertices.get((t + 2) % _length).getY() - vertices.get((t + 1) % _length).getY();
            // double dx2 = vertices.get(t).getX() - vertices.get((t + 1) % _length).getX();
            // double dy2 = vertices.get(t).getY() - vertices.get((t + 1) % _length).getY();
            // double cross = ((dx1 * dy2) - (dy1 * dx2));
            // if (t == 0) {
            //     control = cross > 0;
            // } else {
            //     if (control != (cross > 0)) {
            //         return false;
            //     }
            // }
            final Point2D Qx = _vertices.get((t + 2) % _length);
            final Point2D Qy = _vertices.get((t + 1) % _length);
            final Point2D Qz = _vertices.get(t);

            final Vec2d _x = Qx.toVector();
            final Vec2d _y = Qy.toVector();
            final Vec2d _z = Qz.toVector();

            final Vec2d _diff0 = _x.sub(_y);
            final Vec2d _diff1 = _z.sub(_y);

            final Point2D Wx = new ImmutablePoint2D(_diff0.getX(), _diff0.getY());
            final Point2D Wy = new ImmutablePoint2D(_diff1.getX(), _diff1.getY());

            final double _crossP = ((Wx.getX() * Wy.getY()) - (Wx.getY() * Wy.getX()));
            if (Ints.isEqual(t, 0)) {
                control = Doubles.isGreater(_crossP, 0d);
            } else {
                if (control != Doubles.isGreater(_crossP, 0d)) {
                    return false;
                }
            }

        }

        return true;
    }

    /**
     * @param Polygon2D
     * @param Polygon2D
     * @return
     */
    protected Polygon2D differenceOf(Polygon2D _poly0, Polygon2D _poly1) {
        if (!Ints.isEqual(_poly0.getVerticesCount(), _poly1.getVerticesCount())) {
            throw new IllegalStateException("dimensions are not fit.");
        }

        final List<Point2D> _verts = new ArrayList<>();
        for ( int i = 0; i < _poly0.getVerticesCount(); i++ ) {
            final double _letX = _poly0.getVertices().get(i).getX() - _poly1.getVertices().get(i).getX();
            final double _letY = _poly0.getVertices().get(i).getY() - _poly1.getVertices().get(i).getY();

            _verts.add(new ImmutablePoint2D(_letX, _letY));
        }

        return new ImmutablePolygon2D(_verts);
    }

    @Override
    public Polygon2D diffrence(Polygon2D polygon2D) {
        return differenceOf(this, polygon2D);
    }

    @Override
    public boolean isClosed() {
        final int _length = getVerticesCount();

        if (_length == 0) {
            return true;
        }

        final Point2D _origin = _vertices.get(0);
        final Point2D _end = _vertices.get(_length - 1);

        return _origin.equals(_end);
    }

    /**
     * @param Polygon2D
     * @return
     */
    protected double areaOf(Polygon2D _poly0) {
        final int _length = _poly0.getVerticesCount();
        double _let0 = 0.0;

        Point2D _lastPoint = _poly0.getVertices().get(_length - 1);
        for ( Point2D each : _poly0.getVertices() ) {
            _let0 += (_lastPoint.getX() * each.getY()) - (_lastPoint.getY() * each.getX());
            _lastPoint = each;
        }

        return _let0;
    }

    @Override
    public double area() {
        return areaOf(this);
    }

    @Override
    public Point2D at(int index) {
        if (Ints.isGreater(index, getVerticesCount())) {
            return null;
        }

        return _vertices.get(index);
    }

    @Override
    public int getVerticesCount() {
        if (Objects.isNull(_vertices) || _vertices.isEmpty()) {
            return 0;
        }

        return _vertices.size();
    }

    @Override
    public boolean isBounded() {
        return true;
    }

    @Override
    public List<Point2D> getBounds() {
        final double _greatestX = _vertices.get(_findMaxOrMin(true, true)).getX();
        final double _greatestY = _vertices.get(_findMaxOrMin(false, true)).getY();
        final double _smallestX = _vertices.get(_findMaxOrMin(true, false)).getX();
        final double _smallestY = _vertices.get(_findMaxOrMin(false, false)).getY();

        Point2D _lowerBound = new ImmutablePoint2D(_smallestX, _smallestY);
        Point2D _upperBound = new ImmutablePoint2D(_greatestX, _greatestY);

        return Arrays.asList(_lowerBound, _upperBound);
    }

    @Override
    public DimensionUnit getDimensionUnit() {
        return DimensionUnit.TWO;
    }

    @Override
    public ShapeUnit getShapeUnit() {
        return ShapeUnit.POLYGON;
    }

    @Override
    public Polygon2D addVertex(Point2D point2D) {
        final List<Point2D> _vertx = new ArrayList<>(getVertices());
        _vertx.add(point2D);

        return new ImmutablePolygon2D(_vertx);
    }

    @Override
    public Polygon2D removeVertex(Point2D point2D) {
        final List<Point2D> _vertx = new ArrayList<>(getVertices());
        _vertx.remove(point2D);

        return new ImmutablePolygon2D(_vertx);
    }

    @Override
    public Polygon2D close() {
        if (isClosed() || _vertices.isEmpty()) {
            return this;
        }

        Point2D _origin = getVertices().get(0);
        List<Point2D> _vertx = new ArrayList<>(getVertices());
        _vertx.add(_origin);

        return new ImmutablePolygon2D(_vertx);
    }

    @Override
    public Polygon2D open() {
        if (!isClosed() || _vertices.isEmpty()) {
            return this;
        }

        final int _length = getVerticesCount();
        List<Point2D> _vertx = new ArrayList<>(_vertices);
        _vertx.remove(_length - 1);

        return new ImmutablePolygon2D(_vertx);
    }

    @Override
    public List<Point2D> getVertices() {
        return new ArrayList<>(_vertices);
    }

    @Override
    public ImmutablePolygon2D copy() {
        return new ImmutablePolygon2D(getVertices());
    }

    @Override
    public Polygon2D translate(double... args) {
        if (Objects.isNull(_vertices) || _vertices.isEmpty()) {
            return this;
        }

        final List<Point2D> list = new ArrayList<>();
        for ( Point2D current : _vertices ) {
            list.add(current.translate(args));
        }

        return new ImmutablePolygon2D(list);
    }

    @Override
    public Polygon2D scale(double scaleFactor) {
        if (Objects.isNull(_vertices) || _vertices.isEmpty()) {
            return this;
        }

        final List<Point2D> list = new ArrayList<>();
        for ( Point2D current : _vertices ) {
            list.add(current.scale(scaleFactor));
        }

        return new ImmutablePolygon2D(list);
    }

    @Override
    public int hashCode() {
        int hash = 5;
        hash = 23 * hash + java.util.Objects.hashCode(this._vertices);
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final ImmutablePolygon2D other = (ImmutablePolygon2D) obj;
        if (!java.util.Objects.equals(this._vertices, other._vertices)) {
            return false;
        }
        return true;
    }
}
