/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-geometry project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.geometry.impl;

import java.io.Serializable;
import static java.lang.Math.abs;
import java.util.Arrays;
import java.util.List;
import static terraml.commons.Doubles.isEqual;
import static terraml.commons.Doubles.isGreater;
import static terraml.commons.Doubles.isSmaller;
import terraml.commons.Objects;
import terraml.commons.annotation.File;
import terraml.commons.math.Angle;
import terraml.commons.math.Vec3d;
import terraml.geometry.Point3D;

/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
@File(
        fileName = "ImmutablePoint3D.java",
        packageName = "terraml.geometry.impl",
        projectName = "terraml-geometry"
)
public class ImmutablePoint3D implements Point3D, Serializable {

    private final double x;
    private final double y;
    private final double z;

    /**
     * @param double
     * @param double
     * @param doubles
     */
    public ImmutablePoint3D(double x, double y, double z) {
        this.x = x;
        this.y = y;
        this.z = z;
    }

    /**
     * @param Point3D
     */
    public ImmutablePoint3D(Point3D point3D) {
        this.x = point3D.getX();
        this.y = point3D.getY();
        this.z = point3D.getZ();
    }

    @Override
    public double cosine(Point3D point3D) {
        final Vec3d _vec0 = toVector();
        final Vec3d _vec1 = point3D.toVector();

        final double _dist = _vec0.distanceSquaredTo(_vec1);
        final double _dn = _vec0.getNorm() * _vec1.getNorm();
        final double _calculation = _dist / _dn;

        if (isEqual(abs(_calculation), 1.0d)) {
            return isEqual(_calculation, 1.0d) ? 1 : -1;
        }

        return _calculation;
    }

    @Override
    public double distanceTo(Point3D point3D) {
        final Vec3d _vec0 = toVector();
        final Vec3d _vec1 = point3D.toVector();

        return _vec0.distanceTo(_vec1);
    }

    @Override
    public double angleTo(Point3D shape) {
        final double _formula = (shape.getX() * x + shape.getY() * y + shape.getZ() * z)
                / Math.sqrt((shape.getX() * shape.getX() + shape.getY() * shape.getY()
                             + shape.getZ() * shape.getZ()) * (x * x + y * y + z * z));

        if (isGreater(_formula, 1.0d)) {
            return Angle.fromDegree(0.0).degree;
        } else if (isSmaller(_formula, -1.0d)) {
            return Angle.fromDegree(180.0).degree;
        }

        return Angle.fromRadian(Math.acos(_formula)).degree;
    }

    @Override
    public boolean isParallel(Point3D _p1) {
        return isEqual(cosine(_p1), 1.0);
    }

    @Override
    public boolean isAntiParallel(Point3D _p1) {
        return isEqual(cosine(_p1), -1.0);
    }

    @Override
    public double getX() {
        return this.x;
    }

    @Override
    public double getY() {
        return this.y;
    }

    @Override
    public double getZ() {
        return this.z;
    }

    @Override
    public double[] toArray() {
        return new double[]{x, y, z};
    }

    @Override
    public Vec3d toVector() {
        return new Vec3d(x, y, z);
    }

    @Override
    public boolean isBounded() {
        return true;
    }

    @Override
    public List<Point3D> getBounds() {
        final ImmutablePoint3D _xBound = new ImmutablePoint3D(x, x, x);
        final ImmutablePoint3D _yBound = new ImmutablePoint3D(y, y, y);
        final ImmutablePoint3D _zBound = new ImmutablePoint3D(z, z, z);

        return Arrays.asList(_xBound, _yBound, _zBound);
    }

    @Override
    public ImmutablePoint3D copy() {
        return new ImmutablePoint3D(x, y, z);
    }

    @Override
    public Point3D translate(double... args) {
        if (Objects.isNull(args) || args.length == 0) {
            return copy();
        } else if (args.length == 1) {
            double newX = getX() + args[0];
            double newY = getY() + args[0];
            double newZ = getZ() + args[0];

            return new ImmutablePoint3D(newX, newY, newZ);
        } else if (args.length == 2) {
            double newX = getX() + args[0];
            double newY = getY() + args[1];

            return new ImmutablePoint3D(newX, newY, getZ());
        }

        double newX = getX() + args[0];
        double newY = getY() + args[1];
        double newZ = getZ() + args[2];

        return new ImmutablePoint3D(newX, newY, newZ);
    }

    @Override
    public Point3D scale(double scaleFactor) {
        double newX = getX() * scaleFactor;
        double newY = getY() * scaleFactor;
        double newZ = getZ() * scaleFactor;

        return new ImmutablePoint3D(newX, newY, newZ);
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 31 * hash + (int) (Double.doubleToLongBits(this.x) ^ (Double.doubleToLongBits(this.x) >>> 32));
        hash = 31 * hash + (int) (Double.doubleToLongBits(this.y) ^ (Double.doubleToLongBits(this.y) >>> 32));
        hash = 31 * hash + (int) (Double.doubleToLongBits(this.z) ^ (Double.doubleToLongBits(this.z) >>> 32));
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final ImmutablePoint3D other = (ImmutablePoint3D) obj;
        if (Double.doubleToLongBits(this.x) != Double.doubleToLongBits(other.x)) {
            return false;
        }
        if (Double.doubleToLongBits(this.y) != Double.doubleToLongBits(other.y)) {
            return false;
        }
        if (Double.doubleToLongBits(this.z) != Double.doubleToLongBits(other.z)) {
            return false;
        }
        return true;
    }
}
