/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-geometry project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.geometry.impl;

import java.io.Serializable;
import static java.lang.Math.abs;
import java.util.Arrays;
import java.util.List;
import static terraml.commons.Doubles.isEqual;
import static terraml.commons.Doubles.isGreater;
import static terraml.commons.Doubles.isGreaterEqual;
import static terraml.commons.Doubles.isSmaller;
import static terraml.commons.Doubles.isSmallerEqual;
import terraml.commons.Objects;
import terraml.commons.annotation.File;
import terraml.commons.math.Vec2d;
import terraml.commons.unit.DirectionUnit;
import terraml.geometry.Point2D;

/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
@File(
        fileName = "ImmutablePoint2D.java",
        packageName = "terraml.geometry.impl",
        projectName = "terraml-geometry"
)
public class ImmutablePoint2D implements Point2D, Serializable {

    private final double x;
    private final double y;

    /**
     * @param double
     * @param double
     */
    public ImmutablePoint2D(double x, double y) {
        this.x = x;
        this.y = y;
    }

    /**
     * @param Point2D
     */
    public ImmutablePoint2D(Point2D point2D) {
        this.x = point2D.getX();
        this.y = point2D.getY();
    }

    @Override
    public int ccw(Point2D _p0, Point2D _p1) {
        final Vec2d _v0 = this.toVector();
        final Vec2d _v1 = _p0.toVector();
        final Vec2d _v2 = _p1.toVector();
        final double _crossP = _v0.getCrossProduct(_v1, _v2);

        if (isSmaller(_crossP, 0d)) {
            return -1;
        } else if (isGreater(_crossP, 0)) {
            return 1;
        }

        return 0;
    }

    @Override
    public boolean isBetween(Point2D _p0, Point2D _p1) {
        if (!isEqual(ccw(_p0, _p1), 0d)) {
            return false;
        }

        final boolean _let0 = isEqual(_p0.getX(), _p1.getX());
        final boolean _let1 = isEqual(_p0.getX(), _p1.getY());
        if (_let0 && _let1) {
            return isEqual(_p0.getX(), getX()) && isEqual(_p0.getY(), getY());
        } else if (!_let0) {
            final boolean _let2 = isSmallerEqual(_p0.getX(), getX()) && isSmallerEqual(getX(), _p1.getX());
            final boolean _let3 = isGreaterEqual(_p0.getX(), getX()) && isGreaterEqual(getX(), _p1.getX());
            return _let2 || _let3;
        }

        final boolean _let4 = isSmallerEqual(_p0.getY(), getY()) && isSmallerEqual(getY(), _p1.getY());
        final boolean _let5 = isGreaterEqual(_p0.getY(), getY()) && isGreaterEqual(getY(), _p1.getY());

        return _let4 || _let5;
    }

    /**
     * @param Point2D
     * @param Point2D
     * @return DirectionUnit. However if given Point2D's are equal, then it
     *         returns null.
     */
    protected DirectionUnit directionTo(Point2D _p0, Point2D _p1) {
        final Point2D difference = new ImmutablePoint2D(_p1.getX() - _p0.getX(), _p1.getY() - _p0.getY());
        final boolean _letX = isEqual(difference.getX(), 0);
        final boolean _letY = isEqual(difference.getY(), 0);

        if (_letX && _letY) {
            return null;
        } else if (_letX) {
            return isGreater(difference.getY(), 0) ? DirectionUnit.NORTH : DirectionUnit.SOUTH;
        } else if (_letY) {
            return isGreater(difference.getX(), 0) ? DirectionUnit.EAST : DirectionUnit.WEST;
        } else {
            final boolean diffY = isGreater(difference.getY(), 0);
            if (isGreater(difference.getX(), 0)) {
                return diffY ? DirectionUnit.NORTH_EAST : DirectionUnit.SOUTH_EAST;
            } else {
                return diffY ? DirectionUnit.NORTH_WEST : DirectionUnit.SOUTH_WEST;
            }
        }
    }

    @Override
    public DirectionUnit directionTo(Point2D point2D) {
        return directionTo(this, point2D);
    }

    /**
     * @param Point2D
     * @param Point2D
     * @param Point2D
     * @return
     */
    protected boolean isCollinearWith(Point2D _thePoint, Point2D _p0, Point2D _p1) {
        final Vec2d _v0 = _thePoint.toVector();
        final Vec2d _v1 = _p0.toVector();
        final Vec2d _v2 = _p1.toVector();

        return isSmaller(abs(_v0.getCrossProduct(_v1, _v2)), 1E-8); // epsilon
    }

    @Override
    public boolean isCollinearWith(Point2D p0, Point2D p1) {
        return isCollinearWith(this, p0, p1);
    }

    @Override
    public Point2D centroidOf(List<Point2D> _ptrList) {
        double _letX = 0;
        double _letY = 0;

        for ( int i = 0; i < _ptrList.size(); i++ ) {
            _letX += _ptrList.get(i).getX();
            _letY += _ptrList.get(i).getY();
        }

        _letX /= _ptrList.size();
        _letY /= _ptrList.size();

        return new ImmutablePoint2D(_letX, _letY);
    }

    @Override
    public double getX() {
        return this.x;
    }

    @Override
    public double getY() {
        return this.y;
    }

    @Override
    public double angleTo(Point2D point2D) {
        final double _dx = point2D.getX() - x;
        final double _dy = point2D.getY() - y;

        double theta = Math.atan2(_dy, _dx);

        return Math.toDegrees(theta);
        /*
         * final double _x = getX();
         * final double _y = getY();
         *
         * final double delta = (_x * x + _y * y) / Math.sqrt((_x * _x + _y * _y) * (x * x + y * y));
         *
         * if (delta > 1.0) {
         * return 0.0;
         * }
         *
         * if (delta < -1.0) {
         * return 180.0;
         * }
         *
         * return Math.toDegrees(Math.acos(delta));
         */
    }

    @Override
    public double distanceTo(Point2D point2D) {
        final Vec2d _vec0 = toVector();
        final Vec2d _vec1 = point2D.toVector();

        return _vec0.distanceTo(_vec1);
    }

    @Override
    public double[] toArray() {
        return new double[]{x, y};
    }

    @Override
    public Vec2d toVector() {
        return new Vec2d(x, y);
    }

    @Override
    public boolean isBounded() {
        return true;
    }

    @Override
    public Point2D setX(double x) {
        return new ImmutablePoint2D(x, getY());
    }

    @Override
    public Point2D setY(double y) {
        return new ImmutablePoint2D(getX(), y);
    }

    @Override
    public Point2D scale(double scaleFactor) {
        return new ImmutablePoint2D(getX() * scaleFactor, getY() * scaleFactor);
    }

    @Override
    public Point2D translate(double... args) {
        if (Objects.isNull(args) || args.length == 0) {
            return copy();
        } else if (args.length == 1) {
            double newX = getX() + args[0];
            double newY = getY() + args[0];

            return new ImmutablePoint2D(newX, newY);
        }

        double newX = getX() + args[0];
        double newY = getY() + args[1];

        return new ImmutablePoint2D(newX, newY);
    }

    @Override
    public List<Point2D> getBounds() {
        return Arrays.asList(new ImmutablePoint2D(x, x), new ImmutablePoint2D(y, y));
    }

    @Override
    public Point2D copy() {
        return new ImmutablePoint2D(getX(), getY());
    }

    @Override
    public int hashCode() {
        int hash = 5;
        hash = 79 * hash + (int) (Double.doubleToLongBits(this.x) ^ (Double.doubleToLongBits(this.x) >>> 32));
        hash = 79 * hash + (int) (Double.doubleToLongBits(this.y) ^ (Double.doubleToLongBits(this.y) >>> 32));
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final ImmutablePoint2D other = (ImmutablePoint2D) obj;
        if (Double.doubleToLongBits(this.x) != Double.doubleToLongBits(other.x)) {
            return false;
        }
        if (Double.doubleToLongBits(this.y) != Double.doubleToLongBits(other.y)) {
            return false;
        }
        return true;
    }
}
