/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-geometry project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.geometry.impl;

import java.io.Serializable;
import static java.lang.Math.cos;
import static java.lang.Math.sin;
import java.util.List;
import static terraml.commons.Doubles.isEqual;
import terraml.commons.Objects;
import terraml.commons.annotation.File;
import terraml.commons.math.Vec2d;
import terraml.commons.math.Vec3d;
import terraml.commons.unit.DimensionUnit;
import terraml.geometry.Plane3D;
import terraml.geometry.ShapeUnit;

/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
@File(
        fileName = "ImmutablePlane3D.java",
        packageName = "terraml.geometry.impl",
        projectName = "terraml-geometry"
)
public class ImmutablePlane3D implements Plane3D, Serializable {

    public final Vec3d _normal;
    public final double _distance; // offset from origin.

    /**
     * @param Vec3d
     * @param double
     */
    public ImmutablePlane3D(Vec3d _normal, double _distance) {
        this._normal = _normal;
        this._distance = _distance;
    }

    /**
     * @param Vec3d
     * @param Vec3d
     * @param Vec3d
     */
    public ImmutablePlane3D(Vec3d v0, Vec3d v1, Vec3d v2) {
        final Vec3d _let0 = v1.sub(v0);
        final Vec3d _let1 = v2.sub(v0);
        final Vec3d _cp = _let0.getCrossProduct(_let1);

        this._normal = new Vec3d(_cp);
        this._distance = -1 * (_cp.getDotProduct(v0));
    }

    /**
     * @param double
     * @param double
     * @return
     */
    private static double _getNorm(double x0, double y0) {
        return new Vec2d(x0, y0).getNorm();
    }

    /**
     * @param double
     * @param double
     * @return
     */
    private static double _getAngle(double x0, double y0) {
        return Math.atan2(x0, y0);
    }

    @Override
    public Plane3D rotate(Vec3d vec3d) {
        double _letX = 0.0d;
        double _letY = 0.0d;
        double _letZ = 0.0d;

        if (!isEqual(_normal.x, 0.0d)) {
            final double _angle = _getAngle(_normal.z, _normal.y);
            final double _norm = _getNorm(_normal.y, _normal.z);
            _letY = _norm * cos(_angle + vec3d.x);
            _letZ = _norm * sin(_angle + vec3d.x);
        }

        if (!isEqual(_normal.y, 0.0d)) {
            final double _angle = _getAngle(_normal.z, _normal.x);
            final double _norm = _getNorm(_normal.x, _normal.z);
            _letX = _norm * cos(_angle + vec3d.y);
            _letZ = _norm * sin(_angle + vec3d.y);
        }

        if (!isEqual(_normal.z, 0.0d)) {
            final double _angle = _getAngle(_normal.x, _normal.y);
            final double _norm = _getNorm(_normal.y, _normal.x);
            _letY = _norm * cos(_angle + vec3d.z);
            _letX = _norm * sin(_angle + vec3d.z);
        }

        return new ImmutablePlane3D(new Vec3d(_letX, _letY, _letZ), _distance);
    }

    @Override
    public double distanceTo(Vec3d vec3d) {
        return _normal.getDotProduct(vec3d) + _distance;
    }

    @Override
    public Plane3D getNormalized() {
        final double _length = _normal.getNorm();
        final Vec3d _n = _normal.getNormalized();
        final double _nd = _distance / _length;

        return new ImmutablePlane3D(_n, _nd);
    }

    @Override
    public Vec3d getNormal() {
        return _normal;
    }

    @Override
    public double getDistance() {
        return this._distance;
    }

    @Override
    public List getBounds() {
        return null;
    }

    @Override
    public boolean isBounded() {
        return false;
    }

    @Override
    public ImmutablePlane3D copy() {
        return new ImmutablePlane3D(_normal, _distance);
    }

    @Override
    public DimensionUnit getDimensionUnit() {
        return DimensionUnit.THREE;
    }

    @Override
    public ShapeUnit getShapeUnit() {
        return ShapeUnit.PLANE;
    }

    @Override
    public Plane3D translate(double... args) {
        if (Objects.isNull(args)) {
            return this;
        }

        if (args.length == 1) {
            Vec3d v = getNormal().translate(args[0]);

            return new ImmutablePlane3D(v, _distance);
        }

        return this;
    }

    @Override
    public Plane3D scale(double scaleFactor) {
        Vec3d v = getNormal().scale(scaleFactor);

        return new ImmutablePlane3D(v, _distance);
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 47 * hash + java.util.Objects.hashCode(this._normal);
        hash = 47 * hash + (int) (Double.doubleToLongBits(this._distance) ^ (Double.doubleToLongBits(this._distance) >>> 32));
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final ImmutablePlane3D other = (ImmutablePlane3D) obj;
        if (Double.doubleToLongBits(this._distance) != Double.doubleToLongBits(other._distance)) {
            return false;
        }
        if (!java.util.Objects.equals(this._normal, other._normal)) {
            return false;
        }
        return true;
    }
}
