/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-geometry project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.geometry.impl;

import java.io.Serializable;
import java.util.List;
import java.util.Objects;
import terraml.commons.annotation.File;
import terraml.commons.math.Vec3d;
import terraml.commons.unit.DimensionUnit;
import terraml.geometry.Line3D;
import terraml.geometry.Point3D;
import terraml.geometry.ShapeUnit;

/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
@File(
        fileName = "ImmutablerLine3D",
        packageName = "terraml.geometry.impl",
        projectName = "terraml-geometry"
)
public class ImmutableLine3D implements Line3D, Serializable {

    private final Point3D _origin;
    private final Vec3d _direction;

    /**
     * @param Point3D
     * @param Vec3d
     */
    public ImmutableLine3D(Point3D _origin, Vec3d _direction) {
        this._origin = _origin;
        this._direction = _direction;
    }

    /**
     * @param Point3D
     * @param Point3D
     */
    public ImmutableLine3D(Point3D _origin, Point3D _toDirection) {
        this(_origin, _toDirection.toVector());
    }

    /**
     * @param Line3D
     */
    public ImmutableLine3D(Line3D line3D) {
        this(line3D.getOrigin(), line3D.getDirection());
    }

    @Override
    public boolean skew(Line3D line3D) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    @Override
    public double distanceTo(Point3D point3D) {
        final Vec3d _v0 = point3D.toVector().sub(_origin.toVector());
        final Vec3d _v1 = _direction.getCrossProduct(_v0);

        return _v1.getNorm() / _direction.getNorm();
    }

    @Override
    public double orthogonalDistanceTo(Point3D point3D) {
        final Vec3d _cp = _direction.getCrossProduct(point3D.toVector());
        final Vec3d _v0 = _cp.sub(_direction);
        final double _let1 = _direction.getNorm();

        return _v0.getNorm() / _let1;
    }

    @Override
    public double project(Point3D point3D) {
        final Vec3d _v0 = point3D.toVector().sub(_origin.toVector());
        final Vec3d _v1 = _direction.getCrossProduct(_v0);

        return _direction.getDotProduct(_v1) / _direction.getNormSquared();
    }

    @Override
    public Point3D point(double scalar) {
        final double _letX = _origin.getX() + scalar * _direction.x;
        final double _letY = _origin.getY() + scalar * _direction.y;
        final double _letZ = _origin.getZ() + scalar * _direction.z;

        return new ImmutablePoint3D(_letX, _letY, _letZ);
    }

    // fix conversions
    @Override
    public boolean isParallel(Line3D line3D) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    @Override
    public Point3D getOrigin() {
        return this._origin;
    }

    @Override
    public Vec3d getDirection() {
        return this._direction;
    }

    @Override
    public List getBounds() {
        return null; // line is not bounded. {consider throwing an exception}
    }

    @Override
    public boolean isBounded() {
        return false;
    }

    @Override
    public ImmutableLine3D copy() {
        return new ImmutableLine3D(_origin, _direction);
    }

    @Override
    public DimensionUnit getDimensionUnit() {
        return DimensionUnit.THREE;
    }

    @Override
    public ShapeUnit getShapeUnit() {
        return ShapeUnit.LINE;
    }

    @Override
    public Line3D translate(double... args) {
        final Point3D p = getOrigin().translate(args);

        return new ImmutableLine3D(p, _direction);
    }

    @Override
    public Line3D scale(double scaleFactor) {
        final Point3D p = getOrigin().scale(scaleFactor);

        return new ImmutableLine3D(p, _direction);
    }

    @Override
    public int hashCode() {
        int hash = 5;
        hash = 53 * hash + Objects.hashCode(this._origin);
        hash = 53 * hash + Objects.hashCode(this._direction);
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final ImmutableLine3D other = (ImmutableLine3D) obj;
        if (!Objects.equals(this._origin, other._origin)) {
            return false;
        }
        if (!Objects.equals(this._direction, other._direction)) {
            return false;
        }
        return true;
    }
}
