/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-geometry project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.geometry.impl;

import java.io.Serializable;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import terraml.commons.annotation.File;
import terraml.commons.unit.DimensionUnit;
import terraml.geometry.Circle2D;
import terraml.geometry.Point2D;
import terraml.geometry.ShapeUnit;

/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
@File(
        fileName = "ImmutableCircle2D.java",
        packageName = "terraml.geometry.impl",
        projectName = "terraml-geometry"
)
public class ImmutableCircle2D implements Circle2D, Serializable {

    private final Point2D _center;
    private final double _radius;

    /**
     * @param Point2D
     * @param double
     */
    public ImmutableCircle2D(Point2D _center, double _radius) {
        this._center = _center;
        this._radius = _radius;
    }

    /**
     * @param double
     * @param double
     * @param boolean
     */
    public ImmutableCircle2D(double x, double y, double _radius) {
        this._center = new ImmutablePoint2D(x, y);
        this._radius = _radius;
    }

    /**
     * @param Circle2D
     */
    public ImmutableCircle2D(Circle2D circle2D) {
        this._center = circle2D.getCenter();
        this._radius = circle2D.getRadius();
    }

    @Override
    public double getArea() {
        return Math.PI * (_radius * _radius);
    }

    @Override
    public double angleTo(Circle2D circle2D) {
        return _center.angleTo(circle2D.getCenter());
    }

    @Override
    public double getWidth() {
        return 2 * _radius;
    }

    @Override
    public double getHeight() {
        return 2 * _radius;
    }

    @Override
    public boolean isBounded() {
        return false;
    }

    @Override
    public List<Point2D> getBounds() {
        final double _swX = _center.getX() - _radius;
        final double _swY = _center.getY() - _radius;
        final double _neX = _center.getX() + _radius;
        final double _neY = _center.getY() + _radius;

        return Arrays.asList(new ImmutablePoint2D(_swX, _swY), new ImmutablePoint2D(_neX, _neY));
    }

    @Override
    public DimensionUnit getDimensionUnit() {
        return DimensionUnit.TWO;
    }

    @Override
    public ShapeUnit getShapeUnit() {
        return ShapeUnit.CIRCLE;
    }

    @Override
    public Point2D getCenter() {
        return new ImmutablePoint2D(_center.getX(), _center.getY());
    }

    @Override
    public double getRadius() {
        return _radius;
    }

    @Override
    public ImmutableCircle2D copy() {
        return new ImmutableCircle2D(_center, _radius);
    }

    @Override
    public Circle2D translate(double... args) {
        Point2D p0 = getCenter().translate(args);

        return new ImmutableCircle2D(p0, _radius);
    }

    @Override
    public Circle2D scale(double scaleFactor) {
        Point2D p0 = getCenter().scale(scaleFactor);

        return new ImmutableCircle2D(p0, _radius);
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 79 * hash + Objects.hashCode(this._center);
        hash = 79 * hash + (int) (Double.doubleToLongBits(this._radius) ^ (Double.doubleToLongBits(this._radius) >>> 32));
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final ImmutableCircle2D other = (ImmutableCircle2D) obj;
        if (Double.doubleToLongBits(this._radius) != Double.doubleToLongBits(other._radius)) {
            return false;
        }
        if (!Objects.equals(this._center, other._center)) {
            return false;
        }
        return true;
    }
}
