/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-geometry project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.geometry.impl;

import java.io.Serializable;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import terraml.commons.annotation.File;
import terraml.commons.math.Interval;
import terraml.commons.unit.DimensionUnit;
import terraml.geometry.BBox3D;
import terraml.geometry.Envelope3D;
import terraml.geometry.Point3D;
import terraml.geometry.ShapeUnit;

/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
@File(
        fileName = "ImmutableBBox3D.java",
        packageName = "terraml.geometry.impl",
        projectName = "terraml-geometry"
)
public class ImmutableBBox3D implements BBox3D, Serializable {

    private final Point3D _lowerBound;
    private final Point3D _upperBound;

    /**
     * @param Point3D
     * @param Point3D
     */
    public ImmutableBBox3D(Point3D _lowerBound, Point3D _upperBound) {
        final Point3D[] _fixedBoundings = _fixBoundings(_lowerBound, _upperBound);

        this._lowerBound = _fixedBoundings[0];
        this._upperBound = _fixedBoundings[1];
    }

    /**
     * @param Point3D
     * @param double
     * @param double
     * @param double
     */
    public ImmutableBBox3D(Point3D _lowerB, double w, double h, double d) {
        this._lowerBound = _lowerB;
        this._upperBound = new ImmutablePoint3D(_lowerB.getX() + w, _lowerB.getY() + h, _lowerB.getZ() + d);
    }

    /**
     * @param Envelope3D
     */
    public ImmutableBBox3D(Envelope3D envelope3D) {
        final List<Point3D> _bound = envelope3D.getBounds();
        final Point3D[] _fixedBoundings = _fixBoundings(_bound.get(0), _bound.get(1));

        this._lowerBound = _fixedBoundings[0];
        this._upperBound = _fixedBoundings[1];
    }

    /**
     * @param Point3D
     * @param Point3D
     * @return
     */
    private static Interval[] _axisIntvals(Point3D p0, Point3D p1) {
        final Interval _xAxisIntval = new Interval(p0.getX(), p1.getX());
        final Interval _yAxisIntval = new Interval(p0.getY(), p1.getY());
        final Interval _zAxisIntval = new Interval(p0.getZ(), p1.getZ());

        return new Interval[]{_xAxisIntval, _yAxisIntval, _zAxisIntval};
    }

    /**
     * @param Point3D
     * @param Point3D
     * @return
     */
    private static Point3D[] _fixBoundings(Point3D p0, Point3D p1) {
        final Interval[] _intvals = _axisIntvals(p0, p1);
        final Interval _xAxisIntval = _intvals[0];
        final Interval _yAxisIntval = _intvals[1];
        final Interval _zAxisIntval = _intvals[2];

        final Point3D _low = new ImmutablePoint3D(_xAxisIntval.getLeft(), _yAxisIntval.getLeft(), _zAxisIntval.getLeft());
        final Point3D _upp = new ImmutablePoint3D(_xAxisIntval.getRight(), _yAxisIntval.getRight(), _zAxisIntval.getRight());

        return new Point3D[]{_low, _upp};
    }

    @Override
    public BBox3D union(BBox3D bBox3D) {
        final List<Point3D> _bounds = bBox3D.getBounds();

        final Interval[] _intvals0 = _axisIntvals(_lowerBound, _bounds.get(0));
        final Interval[] _intvals1 = _axisIntvals(_upperBound, _bounds.get(1));

        final Point3D _low = new ImmutablePoint3D(_intvals0[0].getLeft(), _intvals0[1].getLeft(), _intvals0[2].getLeft());
        final Point3D _upp = new ImmutablePoint3D(_intvals1[0].getRight(), _intvals1[1].getRight(), _intvals1[2].getRight());

        return new ImmutableBBox3D(_low, _upp);
    }

    @Override
    public double getVolume() {
        double _w = getWidth();
        double _h = getHeight();
        double _d = getDepth();

        double _const = 1.0;
        return _const * _w * _h * _d;
    }

    @Override
    public Point3D getCentroid() {
        final double _dx = _lowerBound.getX() + (getWidth() * 0.5d);
        final double _dy = _lowerBound.getY() + (getHeight() * 0.5d);
        final double _dz = _lowerBound.getZ() + (getDepth() * 0.5d);

        return new ImmutablePoint3D(_dx, _dy, _dz);
    }

    @Override
    public double getWidth() {
        return _upperBound.getX() - _lowerBound.getX();
    }

    @Override
    public double getHeight() {
        return _upperBound.getY() - _lowerBound.getY();
    }

    @Override
    public double getDepth() {
        return _upperBound.getZ() - _lowerBound.getZ();
    }

    @Override
    public boolean isBounded() {
        return true;
    }

    @Override
    public List<Point3D> getBounds() {
        return Arrays.asList(_lowerBound, _upperBound);
    }

    @Override
    public ImmutableBBox3D copy() {
        return new ImmutableBBox3D(_lowerBound, _upperBound);
    }

    @Override
    public DimensionUnit getDimensionUnit() {
        return DimensionUnit.THREE;
    }

    @Override
    public ShapeUnit getShapeUnit() {
        return ShapeUnit.ENVELOPE;
    }

    @Override
    public BBox3D scale(double scaleFactor) {
        final Point3D p0 = _lowerBound.scale(scaleFactor);
        final Point3D p1 = _upperBound.scale(scaleFactor);

        return new ImmutableBBox3D(p0, p1);
    }

    @Override
    public BBox3D translate(double... args) {
        final Point3D p0 = _lowerBound.translate(args);
        final Point3D p1 = _upperBound.translate(args);

        return new ImmutableBBox3D(p0, p1);
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 53 * hash + Objects.hashCode(this._lowerBound);
        hash = 53 * hash + Objects.hashCode(this._upperBound);
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final ImmutableBBox3D other = (ImmutableBBox3D) obj;
        if (!Objects.equals(this._lowerBound, other._lowerBound)) {
            return false;
        }
        if (!Objects.equals(this._upperBound, other._upperBound)) {
            return false;
        }
        return true;
    }
}
