/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-geometry project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.geometry.impl;

import java.io.Serializable;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import terraml.commons.annotation.File;
import terraml.commons.math.Interval;
import terraml.commons.unit.DimensionUnit;
import terraml.geometry.BBox2D;
import terraml.geometry.Point2D;
import terraml.geometry.ShapeUnit;

/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
@File(
        fileName = "ImmutableBBox2D.java",
        packageName = "terraml.geometry.impl",
        projectName = "terraml-geometry"
)
public class ImmutableBBox2D implements BBox2D, Serializable {

    private final Point2D _southWest;
    private final Point2D _northEast;

    /**
     * @param double
     * @param double
     * @param double
     * @param double
     */
    public ImmutableBBox2D(double x0, double y0, double x1, double y1) {
        this._southWest = new ImmutablePoint2D(x0, y0);
        this._northEast = new ImmutablePoint2D(x1, y1);
    }

    /**
     * @param Point2D
     * @param Point2D
     */
    public ImmutableBBox2D(Point2D _southWest, Point2D _northEast) {
        this._southWest = _southWest;
        this._northEast = _northEast;
    }

    /**
     * @param BBox2D
     */
    public ImmutableBBox2D(BBox2D bBox2D) {
        this._southWest = bBox2D.getSouthWest();
        this._northEast = bBox2D.getNorthEast();
    }

    /**
     * @param Point2D
     * @param Point2D
     * @param boolean
     * @return
     */
    protected final Point2D[] _fixBoundings(Point2D p0, Point2D p1) {
        final Interval _xAxisIntval = new Interval(p0.getX(), p1.getX());
        final Interval _yAxisIntval = new Interval(p0.getY(), p1.getY());

        Point2D _sw = new ImmutablePoint2D(_xAxisIntval.getLeft(), _yAxisIntval.getLeft());
        Point2D _ne = new ImmutablePoint2D(_xAxisIntval.getRight(), _yAxisIntval.getRight());

        return new Point2D[]{_sw, _ne};
    }

    @Override
    public ImmutableBBox2D scale(double scaleFactor) {
        final Point2D sw = getSouthWest().scale(scaleFactor);
        final Point2D ne = getNorthEast().scale(scaleFactor);

        return new ImmutableBBox2D(sw, ne);
    }

    @Override
    public ImmutableBBox2D translate(double... args) {
        final Point2D sw = getSouthWest().translate(args);
        final Point2D ne = getNorthEast().translate(args);

        return new ImmutableBBox2D(sw, ne);
    }

    @Override
    public List<Point2D> getBounds() {
        return Arrays.asList(getSouthWest(), getNorthEast());
    }

    @Override
    public ImmutableBBox2D copy() {
        return new ImmutableBBox2D(getSouthWest(), getNorthEast());
    }

    @Override
    public BBox2D setSouthWest(Point2D point2D) {
        return new ImmutableBBox2D(point2D, getNorthEast());
    }

    @Override
    public BBox2D setNorthEast(Point2D point2D) {
        return new ImmutableBBox2D(getSouthWest(), point2D);
    }

    @Override
    public Point2D getNorthWest() {
        return new ImmutablePoint2D(_southWest.getX(), _southWest.getY() + getHeight());
    }

    @Override
    public Point2D getSouthEast() {
        return new ImmutablePoint2D(_southWest.getX() + getWidth(), _southWest.getY());
    }

    @Override
    public Point2D getNorthEast() {
        return new ImmutablePoint2D(_northEast);
    }

    @Override
    public Point2D getSouthWest() {
        return new ImmutablePoint2D(_southWest);
    }

    @Override
    public Point2D getCentroid() {
        final double _dx = _southWest.getX() + (getWidth() * 0.5d);
        final double _dy = _southWest.getY() + (getHeight() * 0.5d);

        return new ImmutablePoint2D(_dx, _dy);
    }

    @Override
    public ShapeUnit getShapeUnit() {
        return ShapeUnit.ENVELOPE;
    }

    @Override
    public DimensionUnit getDimensionUnit() {
        return DimensionUnit.TWO;
    }

    @Override
    public boolean isBounded() {
        return true;
    }

    @Override
    public double getArea() {
        return getWidth() * getHeight();
    }

    @Override
    public double getHeight() {
        return _northEast.getY() - _southWest.getY();
    }

    @Override
    public double getWidth() {
        return _northEast.getX() - _southWest.getX();
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 67 * hash + Objects.hashCode(this._southWest);
        hash = 67 * hash + Objects.hashCode(this._northEast);
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final ImmutableBBox2D other = (ImmutableBBox2D) obj;
        if (!Objects.equals(this._southWest, other._southWest)) {
            return false;
        }
        if (!Objects.equals(this._northEast, other._northEast)) {
            return false;
        }
        return true;
    }
}
