/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-geometry project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.geometry;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import terraml.commons.Ints;
import static terraml.commons.Ints.isEqual;
import terraml.commons.annotation.File;
import terraml.commons.math.Vec2d;
import terraml.commons.math.Vec3d;
import terraml.geometry.impl.ImmutableBBox2D;
import terraml.geometry.impl.ImmutableBBox3D;
import terraml.geometry.impl.ImmutableCircle2D;
import terraml.geometry.impl.ImmutableLine3D;
import terraml.geometry.impl.ImmutablePlane3D;
import terraml.geometry.impl.ImmutablePoint2D;
import terraml.geometry.impl.ImmutablePoint3D;
import terraml.geometry.impl.ImmutablePolygon2D;
import terraml.geometry.impl.ImmutableSegment2D;
import terraml.geometry.impl.ImmutableSegment3D;
import terraml.geometry.impl.ImmutableSphere3D;

/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
@File(
        fileName = "ShapeFactory.java",
        packageName = "terraml.geometry",
        projectName = "terraml-geometry"
)
public final class ShapeFactory {

    private ShapeFactory() {
    }

    /**
     * @param x
     * @param y
     * @return
     */
    public static ImmutablePoint2D newImmutablePoint2D(double x, double y) {
        return new ImmutablePoint2D(x, y);
    }

    /**
     * @param vec2d
     * @return
     */
    public static ImmutablePoint2D newImmutablePoint2D(Vec2d vec2d) {
        return new ImmutablePoint2D(vec2d.getX(), vec2d.getY());
    }

    /**
     * @param array {acceptable format {x, y}
     * @return
     */
    public static final ImmutablePoint2D newImmutablePoint2D(double[] array) {
        if (isEqual(array.length, 2)) {
            final double[] _refArr = array;
            final double _x = _refArr[0];
            final double _y = _refArr[1];

            return new ImmutablePoint2D(_x, _y);
        }

        throw new IllegalArgumentException("{acceptable format {x, y}");
    }

    /**
     * @param x
     * @param y
     * @return
     */
    public static ImmutablePoint3D newImmutablePoint3D(double x, double y, double z) {
        return new ImmutablePoint3D(x, y, z);
    }

    /**
     * @param vec3d
     * @return
     */
    public static ImmutablePoint3D newImmutablePoint3D(Vec3d vec3d) {
        return new ImmutablePoint3D(vec3d.getX(), vec3d.getY(), vec3d.getZ());
    }

    /**
     * @param array acceptable format: {x, y, z}
     * @return
     */
    public static final ImmutablePoint3D newImmutablePoint3D(double[] array) {
        if (isEqual(array.length, 3)) {
            final double[] _refArr = array;

            final double _x = _refArr[0];
            final double _y = _refArr[1];
            final double _z = _refArr[2];

            return new ImmutablePoint3D(_x, _y, _z);
        }

        throw new IllegalArgumentException("acceptable format: {x, y, z}");
    }

    /**
     * @param sw
     * @param ne
     * @return
     */
    public static final ImmutableBBox2D newImmutableBBox2D(Point2D sw, Point2D ne) {
        return new ImmutableBBox2D(sw, ne);
    }

    /**
     * @param array acceptable format: {x0, y0, x1, y1}
     * @return
     */
    public static final ImmutableBBox2D newImmutableBBox2D(double[] array) {
        if (isEqual(array.length, 4)) {
            final double[] arr = array;

            final Point2D p0 = new ImmutablePoint2D(arr[0], arr[1]);
            final Point2D p1 = new ImmutablePoint2D(arr[2], arr[3]);

            return new ImmutableBBox2D(p0, p1);
        }

        throw new IllegalArgumentException("acceptable format: {x0, y0, x1, y1}");
    }

    /**
     * @param array acceptable format: [{x0, y0}, {x1, y1}]
     * @return
     */
    public static final ImmutableBBox2D newImmutableBBox2D(double[][] array) {
        if (isEqual(array.length, 2)) {
            final double[][] arr = array;

            if (isEqual(arr[0].length, 2) && isEqual(arr[1].length, 2)) {
                final Point2D p0 = new ImmutablePoint2D(arr[0][0], arr[0][1]);
                final Point2D p1 = new ImmutablePoint2D(arr[1][0], arr[1][1]);

                return new ImmutableBBox2D(p0, p1);
            }
        }

        throw new IllegalArgumentException("acceptable format: [{x0, y0}, {x1, y1}]");
    }

    /**
     * @param center
     * @param r
     * @return
     */
    public static final ImmutableCircle2D newImmutableCircle2D(Point2D center, double r) {
        return new ImmutableCircle2D(center, r);
    }

    /**
     * @param array acceptable format: {x, y, radius}
     * @return
     */
    public static final ImmutableCircle2D newImmutableCircle2D(double[] array) {
        if (isEqual(array.length, 3)) {
            final double[] _refArr = array;

            final Point2D _center = new ImmutablePoint2D(_refArr[0], _refArr[1]);
            final double _radius = _refArr[2];

            return new ImmutableCircle2D(_center, _radius);
        }

        throw new IllegalArgumentException("acceptable format: {x, y, radius}");
    }

    /**
     * @param source
     * @param target
     * @return
     */
    public static final ImmutableSegment2D newImmutableSegment2D(Point2D source, Point2D target) {
        return new ImmutableSegment2D(source, target);
    }

    /**
     * @param array acceptable format: {x0, y0, x1, y1}
     * @return
     */
    public static final ImmutableSegment2D newImmutableSegment2D(double[] array) {
        if (isEqual(array.length, 4)) {
            final double[] _refArr = array;

            final Point2D _src = new ImmutablePoint2D(_refArr[0], _refArr[1]);
            final Point2D _trg = new ImmutablePoint2D(_refArr[2], _refArr[3]);

            return new ImmutableSegment2D(_src, _trg);
        }

        throw new IllegalArgumentException("acceptable format: {x0, y0, x1, y1}");
    }

    /**
     * @param array acceptable format: [{x0, y0}, {x1, y1}]
     * @return
     */
    public static final ImmutableSegment2D newImmutableSegment2D(double[][] array) {
        if (isEqual(array.length, 2)) {
            final double[][] _refArr = array;

            if (isEqual(_refArr[0].length, 2) && isEqual(_refArr[1].length, 2)) {
                final Point2D _src = new ImmutablePoint2D(_refArr[0][0], _refArr[0][1]);
                final Point2D _trg = new ImmutablePoint2D(_refArr[1][0], _refArr[1][1]);

                return new ImmutableSegment2D(_src, _trg);
            }
        }

        throw new IllegalArgumentException("acceptable format: [{x0, y0}, {x1, y1}]");
    }

    /**
     * @param list
     * @return
     */
    public static final ImmutablePolygon2D newImmutablePolygon2D(List<Point2D> list) {
        return new ImmutablePolygon2D(list);
    }

    /**
     * @param list
     * @return
     */
    public static final ImmutablePolygon2D newImmutablePolygon2D(Point2D... list) {
        return new ImmutablePolygon2D(Arrays.asList(list));
    }

    /**
     * @param list acceptable format: [{x0, y0}, {x1, y1} ..., {xn, yn}]
     * @return
     */
    public static final ImmutablePolygon2D newImmutablePolygon2D(double[][] array) {
        if (Ints.isEqual(array.length, 4)) {
            final List<Point2D> newVertices = new ArrayList<>();
            for ( int i = 0; i < 4; i++ ) {
                final double[] current = array[i];
                if (!isEqual(current.length, 2)) {
                    throw new IllegalArgumentException("acceptable format: [{x0, y0}, {x1, y1} ..., {xn, yn}]");
                }

                newVertices.add(new ImmutablePoint2D(current[0], current[1]));
            }

            return new ImmutablePolygon2D(newVertices);
        }

        throw new IllegalArgumentException("acceptable format: [{x0, y0}, {x1, y1} ..., {xn, yn}]");
    }

    /**
     * @param p0
     * @param p1
     * @return
     */
    public static final ImmutableBBox3D newImmutableBBox3D(Point3D p0, Point3D p1) {
        return new ImmutableBBox3D(p0, p1);
    }

    /**
     * @param p0
     * @param w
     * @param h
     * @param d
     * @return
     */
    public static final ImmutableBBox3D newImmutableBBox3D(Point3D p0, double w, double h, double d) {
        return new ImmutableBBox3D(p0, w, h, d);
    }

    /**
     * @param array acceptable format: [{x0, y0, z0}, {x1, y1, z1} ]
     * @return
     */
    public static final ImmutableBBox3D newImmutableBBox3D(double[][] array) {
        if (Ints.isEqual(array.length, 2)) {
            final double[] _p0 = array[0];
            final double[] _p1 = array[1];

            if (Ints.isEqual(_p0.length, 3) && Ints.isEqual(_p1.length, 3)) {
                final Point3D _low = new ImmutablePoint3D(_p0[0], _p0[1], _p0[2]);
                final Point3D _upp = new ImmutablePoint3D(_p1[0], _p1[1], _p1[2]);

                return new ImmutableBBox3D(_low, _upp);
            }
        }

        throw new IllegalArgumentException("acceptable format: [{x0, y0, z0}, {x1, y1, z1}]");
    }

    /**
     * @param point3D
     * @param vec3d
     * @return
     */
    public static final ImmutableLine3D newImmutableLine3D(Point3D point3D, Vec3d vec3d) {
        return new ImmutableLine3D(point3D, vec3d);
    }

    /**
     * @param point3D
     * @param dirr
     * @return
     */
    public static final ImmutableLine3D newImmutableLine3D(Point3D point3D, Point3D dirr) {
        return new ImmutableLine3D(point3D, dirr);
    }

    /**
     * @param array acceptable format: [{x0, y0, z0}, {x1, y1, z1}]
     * @return
     */
    public static final ImmutableLine3D newImmutableLine3D(double[][] array) {
        if (Ints.isEqual(array.length, 2)) {
            final double[] _p0 = array[0];
            final double[] _p1 = array[1];

            if (Ints.isEqual(_p0.length, 3) && Ints.isEqual(_p1.length, 3)) {
                final Point3D _low = new ImmutablePoint3D(_p0[0], _p0[1], _p0[2]);
                final Vec3d _dirr = new Vec3d(_p1[0], _p1[1], _p1[2]);

                return new ImmutableLine3D(_low, _dirr);
            }
        }

        throw new IllegalArgumentException("acceptable format: [{x0, y0, z0}, {x1, y1, z1}]");
    }

    /**
     * @param vec3d
     * @param offset
     * @return
     */
    public static final ImmutablePlane3D newImmutablePlane3D(Vec3d vec3d, double offset) {
        return new ImmutablePlane3D(vec3d, offset);
    }

    /**
     * @param v0
     * @param v1
     * @param v2
     * @return
     */
    public static final ImmutablePlane3D newImmutablePlane3D(Vec3d v0, Vec3d v1, Vec3d v2) {
        return new ImmutablePlane3D(v0, v1, v2);
    }

    /**
     * @param src
     * @param tar
     * @return
     */
    public static final ImmutableSegment3D newImmutableSegment3D(Point3D src, Point3D tar) {
        return new ImmutableSegment3D(src, tar);
    }

    /**
     * @param array acceptable format: [{x0, y0, z0}, {x1, y1, z1}]
     * @return
     */
    public static final ImmutableSegment3D newImmutableSegment3D(double[][] array) {
        if (Ints.isEqual(array.length, 2)) {
            final double[] _p0 = array[0];
            final double[] _p1 = array[1];

            if (Ints.isEqual(_p0.length, 3) && Ints.isEqual(_p1.length, 3)) {
                final Point3D _src = new ImmutablePoint3D(_p0[0], _p0[1], _p0[2]);
                final Point3D _tar = new ImmutablePoint3D(_p1[0], _p1[1], _p1[2]);

                return new ImmutableSegment3D(_src, _tar);
            }
        }

        throw new IllegalArgumentException("acceptable format: [{x0, y0, z0}, {x1, y1, z1}]");
    }

    /**
     * @param center
     * @param radius
     * @return
     */
    public static final ImmutableSphere3D newImmutableSphere3D(Point3D center, double radius) {
        return new ImmutableSphere3D(center, radius);
    }
}
