/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-geometry project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.geometry;

import static java.lang.Math.abs;
import java.util.Arrays;
import java.util.List;
import terraml.commons.Doubles;
import static terraml.commons.Doubles.isGreater;
import static terraml.commons.Doubles.isGreaterEqual;
import static terraml.commons.Doubles.isSmaller;
import static terraml.commons.Doubles.isSmallerEqual;
import terraml.commons.annotation.File;
import terraml.commons.annotation.Issue;
import terraml.commons.annotation.Reference;
import terraml.commons.math.Interval;
import terraml.commons.math.Vec2d;
import terraml.commons.math.Vec3d;
import terraml.geometry.impl.ImmutablePoint2D;
import terraml.geometry.impl.ImmutableSegment2D;

/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
@File(
        fileName = "Intersector.java",
        packageName = "terraml.geometry",
        projectName = "terraml-geometry"
)
public final class Intersector {

    private Intersector() {
    }

    /**
     * @param Polygon2D
     * @param Point2D
     * @return true if given Polygon2D contains given Point2D. Otherwise false.
     */
    public static boolean contains(Polygon2D _container, Point2D _p0) {
        Point2D _currentP = _container.getVertices().get(0);
        final int _length = _container.getVerticesCount();

        int _rayCross = 0;
        for ( int t = 1; t < _length; t++ ) {
            boolean _dy0 = _currentP.getY() <= _p0.getY() && _p0.getY() < _container.getVertices().get(t).getY();
            boolean _dy1 = _container.getVertices().get(t).getY() <= _p0.getY() && _p0.getY() < _currentP.getY();

            if (_dy0 || _dy1) {
                double crs = (_container.getVertices().get(t).getX() - _currentP.getX()) * (_p0.getY() - _currentP.getY())
                        / (_container.getVertices().get(t).getY() - _currentP.getY()) + _currentP.getX();
                if (_p0.getX() < crs) {
                    _rayCross++;
                }
            }

            _currentP = _container.getVertices().get(t);
        }

        return (_rayCross % 2) == 1;
    }

    /**
     * @param Circle2D
     * @param Point2D
     * @return true if given Circle2D contains given Point2D. Otherwise false.
     *         However, if given Point2D is on top of Circle2D, method doesn't consider
     *         Point2D is contained. In that case method returns false.
     */
    public static boolean contains(Circle2D _container, Point2D _p0) {
        final Vec2d _v0 = _container.getCenter().toVector();
        final Vec2d _v1 = _p0.toVector();

        final double _letX = _v0.distanceTo(_v1);

        return isSmallerEqual(_letX, _container.getRadius());
    }

    /**
     * @param Envelope2D
     * @param Point2D
     * @return true if given Envelope2D contains given Point2D. Otherwise false.
     *         if given Point2D is on top of Envelope2D, then method consider Point2D is
     *         contained.
     */
    public static boolean contains(Envelope2D _container, Point2D _point2) {
        final Interval _intvalX = getXInterval(_container);
        final Interval _intvalY = getYInterval(_container);

        return _intvalX.inRange(_point2.getX()) && _intvalY.inRange(_point2.getY());
    }

    /**
     * @param Envelope3D
     * @param Point3D
     * @return
     */
    public static boolean contains(Envelope3D _container, Point3D _point3) {
        final Interval _intvalX = getXInterval(_container);
        final Interval _intvalY = getYInterval(_container);
        final Interval _intvalZ = getZInterval(_container);

        return _intvalX.inRange(_point3.getX())
                && _intvalY.inRange(_point3.getY())
                && _intvalZ.inRange(_point3.getZ());
    }

    /**
     * @param Sphere3D
     * @param Point3D
     * @return
     */
    public static boolean contains(Sphere3D _container, Point3D _p0) {
        return isSmallerEqual(_container.getCenter().distanceTo(_p0), _container.getRadius());
    }

    /**
     * @param Line3D
     * @param Point3D
     * @return
     */
    public static Vec3d nearestPoint(Line3D _container, Point3D _p0) {
        final Vec3d _v0 = _p0.toVector();
        final Vec3d _v1 = _container.getOrigin().toVector();
        final Vec3d _v2 = _v0.sub(_v1);

        double _let0 = _v2.getDotProduct(_container.getDirection());
        double _let1 = _container.getDirection().getNorm();

        final Vec3d _v3 = _container.getDirection().scale(_let0 / _let1);

        return _v1.add(_v3);
    }

    /**
     * @see com.terraml.geometry.Intersector.contains(Circle2D _container,
     * Circle2D _circle2)
     * @see com.terraml.geometry.Intersector.within(Circle2D _circle2, Circle2D
     * _container)
     *
     * @param Circle2D
     * @param Circle2D
     * @return
     */
    private static boolean _overlap(Circle2D _container, Circle2D _circle2) {
        final Vec2d _v0 = _container.getCenter().toVector();
        final Vec2d _v1 = _circle2.getCenter().toVector();

        final double _let0 = _v0.distanceTo(_v1);
        final double _let1 = _container.getRadius() - _circle2.getRadius();

        return isSmallerEqual(_let0, _let1);
    }

    /**
     * @see com.terraml.geometry.Intersector._overlap(Circle2D _container,
     * Circle2D _circle2)
     *
     * @param Circle2D
     * @param Circle2D
     * @return true if given first Circle2D contains second Circle2D.
     */
    public static boolean contains(Circle2D _container, Circle2D _circle2) {
        return _overlap(_container, _circle2);
    }

    /**
     * @see com.terraml.geometry.Intersector._overlap(Circle2D _container,
     * Circle2D _circle2)
     *
     * @param Circle2D
     * @param Circle2D
     * @return true if given first Circle2D contained in the second Circle2D.
     */
    public static boolean within(Circle2D _circle2, Circle2D _container) {
        return _overlap(_container, _circle2);
    }

    /**
     * @see com.terraml.geometry.Intersector.intersects(Circle2D shape0,
     * Circle2D shape1)
     *
     * @param Circle2D
     * @param Circle2D
     * @return
     */
    private static boolean _intersects(Circle2D shape0, Circle2D shape1) {
        if (_overlap(shape0, shape1) || _overlap(shape1, shape0)) {
            return false;
        }

        final Vec2d _v0 = shape0.getCenter().toVector();
        final Vec2d _v1 = shape1.getCenter().toVector();

        final double _let0 = _v0.distanceTo(_v1);
        final double _let1 = shape0.getRadius() + shape1.getRadius();

        return isSmallerEqual(_let0, _let1);
    }

    /**
     * @see com.terraml.geometry.Intersector._intersects(Circle2D shape0,
     * Circle2D shape1)
     *
     * @param Circle2D
     * @param Circle2D
     * @return true if given first Circle2D intersects with second Circle2D
     */
    public static boolean intersects(Circle2D shape0, Circle2D shape1) {
        return _intersects(shape0, shape1);
    }

    /**
     * @see com.terraml.geometry.Intersector.contains(Circle2D containerShape,
     * Envelope2D shape)
     * @see com.terraml.geometry.Intersector.contains(Circle2D _container,
     * Point2D _p0)
     *
     * @param _container
     * @param _env2
     * @return
     */
    private static boolean _contains(Circle2D _container, Envelope2D _env2) {
        if (contains(_container, _env2.getCentroid())) {
            return true;
        }

        boolean _allBoundsIn = true;
        final List<Point2D> _cornerList = exportCorners(_env2);

        for ( Point2D _each : _cornerList ) {
            if (!contains(_container, _each)) {
                _allBoundsIn = false;
                break;
            }
        }

        return _allBoundsIn;
    }

    /**
     * @see com.terraml.geometry.Intersector._contains(Circle2D containerShape,
     * Envelope2D shape)
     *
     * @param containerShape
     * @param shape
     * @return true if given Circle2D contains given Envelope2D. Otherwise
     *         false.
     */
    public static boolean contains(Circle2D containerShape, Envelope2D shape) {
        return _contains(containerShape, shape);
    }

    /**
     * @see com.terraml.geometry.Intersector.contains(Circle2D _container,
     * Segment2D _segment2)
     * @see com.terraml.geometry.Intersector.contains(Circle2D _container,
     * Point2D _p0)
     *
     * @param Circle2D
     * @param Segment2D
     * @return
     */
    private static boolean _contains(Circle2D _container, Segment2D _segment2) {
        final boolean _let0 = contains(_container, _segment2.getSource());
        if (!_let0) {
            return false;
        }
        final boolean _let1 = contains(_container, _segment2.getTarget());

        return _let0 && _let1;
    }

    /**
     * @see com.terraml.geometry.Intersector._contains(Circle2D _container,
     * Segment2D _segment2)
     *
     * @param Circle2D
     * @param Segment2D
     * @return true if given Circle2D is contains given Segment2D. Otherwise
     *         false.
     */
    public static boolean contains(Circle2D _container, Segment2D _segment2) {
        return _contains(_container, _segment2);
    }

    /**
     * @see com.terraml.geometry.Intersector.contains(Envelope2D _container,
     * Circle2D _circle2)
     * @see com.terraml.geometry.Intersector.contains(Envelope2D _container,
     * Point2D _point2)
     *
     * @param Envelope2D
     * @param Circle2D
     * @return
     */
    @Issue(
            category = Issue.Type.BUG,
            content = "algorithm",
            isFixed = false,
            issueId = "TMLGEOM-1"
    )
    private static boolean _contains(Envelope2D _container, Circle2D _circle2) {
        return _circle2.getBounds()
                .stream()
                .noneMatch((_eachB) -> (!contains(_container, _eachB)));
    }

    /**
     * @see com.terraml.geometry.Intersector._contains(Envelope2D _container,
     * Circle2D _circle2)
     *
     * @param Envelope2D
     * @param Circle2D
     * @return true if given Envelope2D contains Circle2D. Otherwise false.
     */
    public static boolean contains(Envelope2D _container, Circle2D _circle2) {
        return _contains(_container, _circle2);
    }

    /**
     * @param double
     * @param double
     * @param double
     * @return
     */
    private static double _determine(double _ax, double _ls, double _gr) {
        if (Doubles.isSmaller(_ax, _ls)) {
            return _ls;
        } else if (Doubles.isGreater(_ax, _gr)) {
            return _gr;
        }

        return _ax;
    }

    /**
     * @see com.terraml.geometry.Intersector.intersects(Envelope2D shape0,
     * Circle2D shape1)
     *
     * @param Envelope2D
     * @param Circle2D
     * @return
     */
    private static boolean _intersects(Envelope2D _env2, Circle2D _circle2) {
        double _letX = _circle2.getCenter().getX();
        double _letY = _circle2.getCenter().getY();

        Point2D _lowerBnd = _env2.getSouthWest();
        Point2D _upperBnd = _env2.getNorthEast();

        _letX = _determine(_letX, _lowerBnd.getX(), _upperBnd.getX());
        _letY = _determine(_letY, _lowerBnd.getY(), _upperBnd.getY());

        double _dstX = _letX - _circle2.getCenter().getX();
        double _dstY = _letY - _circle2.getCenter().getY();

        return isSmallerEqual(((_dstX * _dstX) + (_dstY * _dstY)), (_circle2.getRadius() * _circle2.getRadius()));
    }

    /**
     * @see com.terraml.geometry.Intersector._intersects(Envelope2D _env2,
     * Circle2D _circle2)
     *
     * @param Envelope2D
     * @param Circle2D
     * @return true if given Envelope2D and Circle2D are intersects. Otherwise
     *         false.
     */
    public static boolean intersects(Envelope2D shape0, Circle2D shape1) {
        return _intersects(shape0, shape1);
    }

    /**
     * @see com.terraml.geometry.Intersector.contains(Envelope2D containerShape,
     * Envelope2D _env2)
     * @see com.terraml.sharedlib.math.Interval
     *
     * @param Envelope2D
     * @param Envelope2D
     * @return
     */
    private static boolean _contains(Envelope2D _container, Envelope2D _env2) {
        final Interval _env0X = getXInterval(_container);
        final Interval _env0Y = getYInterval(_container);
        final Interval _env1X = getXInterval(_env2);
        final Interval _env1Y = getYInterval(_env2);

        return _env0X.intersects(_env1X) && _env0Y.intersects(_env1Y);
    }

    /**
     * @see com.terraml.geometry.Intersector._contains(Envelope2D _container,
     * Envelope2D _env2)
     *
     * @param Envelope2D
     * @param Envelope2D
     * @return true if given first Envelope2D contains other Envelope2D.
     *         Otherwise false.
     */
    public static boolean contains(Envelope2D containerShape, Envelope2D _env2) {
        return _contains(containerShape, _env2);
    }

    /**
     * @see com.terraml.geometry.Intersector._contains(Envelope2D _container,
     * Envelope2D _env2)
     *
     * @param Envelope2D
     * @param Envelope2D
     * @return true if given second Envelope2D contains first Envelope2D.
     *         Otherwise false.
     */
    public static boolean within(Envelope2D _env2, Envelope2D containerShape) {
        return _contains(containerShape, _env2);
    }

    // check later.
    private static boolean _intersects(Envelope2D _env0, Envelope2D _env1) {
        final Interval _env0X = getXInterval(_env0);
        final Interval _env0Y = getYInterval(_env0);
        final Interval _env1X = getXInterval(_env1);
        final Interval _env1Y = getYInterval(_env1);

        return _env0X.intersects(_env1X) && _env0Y.intersects(_env1Y);
    }

    /**
     * @see com.terraml.geometry.Intersector._intersects(Envelope2D _env0,
     * Envelope2D _env1)
     *
     * @param Envelope2D
     * @param Envelope2D
     * @return true if given Envelope2D's are intersects. Otherwise false.
     */
    public static boolean intersects(Envelope2D _env0, Envelope2D _env1) {
        return _intersects(_env0, _env1);
    }

    /**
     * @see com.terraml.geometry.Intersector.contains(Envelope2D _container,
     * Point2D _point2)
     * @see com.terraml.geometry.Intersector.contains(Envelope2D containerShape,
     * Segment2D shape)
     *
     * @param Envelope2D
     * @param Segment2D
     * @return
     */
    private static boolean _contains(Envelope2D containerShape, Segment2D shape) {
        return contains(containerShape, shape.getSource())
                && contains(containerShape, shape.getTarget());
    }

    /**
     * @see com.terraml.geometry.Intersector._contains(Envelope2D
     * containerShape, Segment2D shape)
     *
     * @param Envelope2D
     * @param Envelope2D
     * @return true if given Envelope2D contains given Segment2D. Otherwise
     *         false.
     */
    public static boolean contains(Envelope2D containerShape, Segment2D shape) {
        return _contains(containerShape, shape);
    }

    /**
     * @see com.terraml.geometry.Intersector.contains(Envelope2D _container,
     * Point2D _point2)
     * @see com.terraml.geometry.Intersector.within(Envelope3D shape, Envelope3D
     * containerShape)
     * @see com.terraml.geometry.Intersector.contains(Envelope3D containerShape,
     * Envelope3D shape)
     *
     * @param Envelope3D
     * @param Envelope3D
     * @return
     */
    private static boolean _contains(Envelope3D _container, Envelope3D _env3) {
        final List<Point3D> _bounds = _env3.getBounds();

        return _bounds.stream().noneMatch((_eachB) -> (!contains(_container, _eachB)));
    }

    /**
     * @see com.terraml.geometry.Intersector._contains(Envelope3D _container,
     * Envelope3D _env3)
     *
     * @param Envelope3D
     * @param Envelope3D
     * @return
     */
    public static boolean within(Envelope3D shape, Envelope3D containerShape) {
        return _contains(containerShape, shape);
    }

    /**
     * @see com.terraml.geometry.Intersector._contains(Envelope3D _container,
     * Envelope3D _env3)
     *
     * @param Envelope3D
     * @param Envelope3D
     * @return
     */
    public static boolean contains(Envelope3D containerShape, Envelope3D shape) {
        return _contains(containerShape, shape);
    }

    /**
     * @see com.terraml.sharedlib.math.Interval
     *
     * @param shape0
     * @param Envelope3D
     * @return
     */
    private static boolean _intersects(Envelope3D shape0, Envelope3D shape1) {
        final Interval[] _intValXYZ0 = getXYZInterval(shape0);
        final Interval[] _intValXYZ1 = getXYZInterval(shape1);

        for ( int i = 0; i < 3; i++ ) {
            if (!_intValXYZ0[i].intersects(_intValXYZ1[i])) {
                return false;
            }
        }

        return false;
    }

    @Reference(
            title = "Seperating Axes Theorem 3D",
            link = "https://www.geometrictools.com/Documentation/DynamicCollisionDetection.pdf"
    )
    private static boolean _intersects2(Envelope3D shape0, Envelope3D shape1) {
        double _letX = shape1.getCentroid().getX() - shape0.getCentroid().getX();
        double _letW = (shape0.getWidth() / 2) + (shape1.getWidth() / 2);

        double _letY = shape1.getCentroid().getY() - shape0.getCentroid().getY();
        double _letH = (shape0.getHeight() / 2) + (shape1.getHeight() / 2);

        double _letZ = shape1.getCentroid().getZ() - shape0.getCentroid().getZ();
        double _letD = (shape0.getDepth() / 2) + (shape1.getDepth() / 2);

        _letX = abs(_letX);
        _letY = abs(_letY);
        _letZ = abs(_letZ);

        return isSmallerEqual(_letX, _letW) && isSmallerEqual(_letY, _letH) && isSmallerEqual(_letZ, _letD);
    }

    /**
     * @see com.terraml.geometry.Intersector._intersects2(Envelope3D shape0,
     * Envelope3D shape1)
     *
     * @param Envelope3D
     * @param Envelope3D
     * @return
     */
    public static boolean intersects(Envelope3D shape0, Envelope3D shape1) {
        return _intersects2(shape0, shape1);
    }

    /**
     * @see com.terraml.geometry.Intersector.contains(Envelope3D _container,
     * Point3D _point3)
     * @see com.terraml.geometry.Intersector.contains(Envelope3D containerShape,
     * Segment3D shape)
     *
     * @param Envelope3D
     * @param Segment3D
     * @return
     */
    private static boolean _contains(Envelope3D containerShape, Segment3D shape) {
        final Point3D[] _srcTar = new Point3D[]{shape.getSource(), shape.getTarget()};

        for ( Point3D _eachP : _srcTar ) {
            if (!contains(containerShape, _eachP)) {
                return false;
            }
        }

        return true;
    }

    /**
     * @see com.terraml.geometry.Intersector._contains(Envelope3D
     * containerShape, Segment3D shape)
     *
     * @param Envelope3D
     * @param Segment3D
     * @return
     */
    public static boolean contains(Envelope3D containerShape, Segment3D shape) {
        return _contains(containerShape, shape);
    }

    /**
     * @see com.terraml.geometry.Intersector.intersects(Segment2D shape0,
     * Circle2D shape1)
     *
     * @param Segment2D
     * @param Circle2D
     * @return
     */
    private static boolean _intersects(Segment2D _seg2, Circle2D _circle2) {
        final Vec2d _v0 = _seg2.getSource().toVector();
        final Vec2d _v1 = _seg2.getTarget().toVector();
        final Vec2d _v2 = _circle2.getCenter().toVector();

        Vec2d _v3 = _v1.sub(_v0);
        Vec2d _v4 = _v2.sub(_v0);

        double _nx;
        double _ny;

        final double _let0 = _v3.getNorm();
        final double _let1 = _v4.getDotProduct(_v3.getNormalized());

        if (isSmallerEqual(_let1, 0)) {
            _nx = _v0.getX();
            _ny = _v0.getY();
        } else if (isGreaterEqual(_let1, _let0)) {
            _nx = _v1.getX();
            _ny = _v1.getY();
        } else {
            Vec2d _v5 = _v0.scale(_let1);

            _nx = _v5.x + _v0.x;
            _ny = _v5.y + _v0.y;
        }

        final double _letX = _v2.x - _nx;
        final double _letY = _v2.y - _ny;
        final double _dst = (_letX * _letX) + (_letY * _letY);

        // sqrt(_circle2.getRadius()); ??
        return isSmallerEqual(_dst, _circle2.getRadius() * _circle2.getRadius());
    }

    /**
     * @see com.terraml.geometry.Intersector._intersects(Segment2D _seg2,
     * Circle2D _circle2)
     *
     * @param Segment2D
     * @param Circle2D
     * @return true if given Segment2D and Circle2D are intersects.
     */
    public static boolean intersects(Segment2D shape0, Circle2D shape1) {
        return _intersects(shape0, shape1);
    }

    @Reference(
            link = "http://stackoverflow.com/questions/99353/how-to-test-if-a-line-segment-intersects-an-axis-aligned-rectange-in-2d",
            implNote = "Implementation is clear but it must be simplified. "
            + "Consider 'Cohen Sutherland Algorithm'. There is a example implementation:"
            + "https://www.devin.com/cruft/cs360/draw/draw/CohenSutherland.java"
    )
    private static boolean _intersects(Segment2D _seg2, Envelope2D _env2) {
        if (contains(_env2, _seg2)) {
            return false;
        }

        double p1x = _seg2.getSource().getX();
        double p2x = _seg2.getTarget().getX();
        double minX = p1x;
        double maxX = p2x;

        if (isGreater(p1x, p2x)) {
            minX = p2x;
            maxX = p1x;
        }

        double rectMinX = _env2.getSouthWest().getX();
        double rectMaxX = _env2.getNorthEast().getX();

        if (isGreater(maxX, rectMaxX)) {
            maxX = rectMaxX;
        }
        if (isSmaller(minX, rectMinX)) {
            minX = rectMinX;
        }
        if (isGreater(minX, maxX)) {
            return false;
        }

        double p1y = _seg2.getSource().getY();
        double p2y = _seg2.getTarget().getY();
        double minY = p1y;
        double maxY = p2y;
        double _dx = p2x - p1x;

        if (isGreater(abs(_dx), 1E-8)) { // epsilon
            double _let0 = (p2y - p1y) / _dx;
            double _let1 = p1y - _let0 * p1x;
            minY = _let0 * minX + _let1;
            maxY = _let0 * maxX + _let1;
        }
        if (isGreater(minY, maxY)) {
            double tmp = maxY;
            maxY = minY;
            minY = tmp;
        }

        double rectMinY = _env2.getSouthWest().getY();
        double rectMaxY = _env2.getNorthEast().getY();

        if (isGreater(maxY, rectMaxY)) {
            maxY = rectMaxY;
        }
        if (isSmaller(minY, rectMinY)) {
            minY = rectMinY;
        }

        return !isGreater(minY, maxY);
    }

    /**
     * @see com.terraml.geometry.Intersector._intersects(Segment2D _seg2,
     * Envelope2D _env2)
     * @param Segment2D
     * @param Envelope2D
     * @return true if given Envelope2D intersects with Segment2D. Otherwise
     *         false.
     */
    public static boolean intersects(Segment2D shape0, Envelope2D shape1) {
        return _intersects(shape0, shape1);
    }

    /**
     * @see com.terraml.geometry.Intersector.intersects(Segment2D shape0,
     * Segment2D shape1)
     *
     * @param Segment2D
     * @param Segment2D
     * @return
     */
    private static boolean _intersects(Segment2D shape0, Segment2D shape1) {
        final double v0x = shape0.getSource().getX();
        final double v0y = shape0.getSource().getY();

        final double v1x = shape0.getTarget().getX();
        final double v1y = shape0.getTarget().getY();

        final double v2x = shape1.getSource().getX();
        final double v2y = shape1.getSource().getY();

        final double v3x = shape1.getTarget().getX();
        final double v3y = shape1.getTarget().getY();

        final double Qx = v1x - v0x;
        final double Qy = v1y - v0y;
        final double Wx = v3x - v2x;
        final double Wy = v3y - v2y;

        final double dx = v0x - v2x;
        final double dy = v0y - v2y;
        final double d = -Wx * Qy + Qx * Wy;

        double Zr = (-Qy * dx + Qx * dy) / d;
        double Zt = (Wx * dy - Wy * dx) / d;

        return Zr >= 0 && Zr <= 1 && Zt >= 0 && Zt <= 1;
    }

    /**
     * @see com.terraml.geometry.Intersector._intersects(Segment2D shape0,
     * Segment2D shape1)
     *
     * @param Segment2D
     * @param Segment2D
     * @return true if given Segment2D's are intersects. Otherwise false.
     */
    public static boolean intersects(Segment2D shape0, Segment2D shape1) {
        return _intersects(shape0, shape1);
    }

    /**
     * @see com.terraml.geometry.Intersector.intersects(Segment3D shape0,
     * Plane3D shape1)
     *
     * @param Segment3D
     * @param Plane3D
     * @return
     */
    private static boolean _intersects(Segment3D _seg3, Plane3D _pl3) {
        Vec3d _v0 = _seg3.getSource().toVector();
        Vec3d _v1 = _seg3.getTarget().toVector();

        Vec3d _v2 = _v1.sub(_v0);

        double _let0 = _v2.getDotProduct(_pl3.getNormal());
        double _let1 = _v0.getDotProduct(_pl3.getNormal()) + _pl3.getDistance();
        double _let2 = -1 * (_let1 / _let0);

        return !(isSmaller(_let2, 0) || isGreater(_let2, 1));
    }

    /**
     * @see com.terraml.geometry.Intersector._intersects(Segment3D _seg3,
     * Plane3D _pl3)
     *
     * @param Segment3D
     * @param Plane3D
     * @return
     */
    public static boolean intersects(Segment3D shape0, Plane3D shape1) {
        return _intersects(shape0, shape1);
    }

    /**
     * @see com.terraml.geometry.Intersector.intersects(Sphere3D shape0, Line3D
     * shape1)
     *
     * @param Sphere3D
     * @param Line3D
     * @return
     */
    private static boolean _intersects(Sphere3D _sph3, Line3D _ln3) {
        final Vec3d _v0 = _sph3.getCenter().toVector();
        final double _let0 = _v0.distanceTo(nearestPoint(_ln3, _sph3.getCenter()));

        return isSmallerEqual(_let0, _sph3.getRadius());
    }

    /**
     * @see com.terraml.geometry.Intersector._intersects(Sphere3D _sph3, Line3D
     * _ln3)
     *
     * @param Sphere3D
     * @param Line3D
     * @return
     */
    public static boolean intersects(Sphere3D shape0, Line3D shape1) {
        return _intersects(shape0, shape1);
    }

    /**
     * @see com.terraml.geometry.Intersector.intersects(Sphere3D shape0, Plane3D
     * shape1)
     *
     * @param Sphere3D
     * @param Plane3D
     * @return
     */
    private static boolean _intersects(Sphere3D _sph3, Plane3D _pla3) {
        final Vec3d _v0 = _sph3.getCenter().toVector();
        final double _let0 = _pla3.getNormal().getDotProduct(_v0);

        return isSmallerEqual(_let0, _sph3.getRadius());
    }

    /**
     * @see com.terraml.geometry.Intersector._intersects(Sphere3D _sph3, Plane3D
     * _pla3)
     *
     * @param Sphere3D
     * @param Plane3D
     * @return
     */
    public static boolean intersects(Sphere3D shape0, Plane3D shape1) {
        return _intersects(shape0, shape1);
    }

    /**
     * @param _env
     * @return segments from envelopes as List. {Not ordered.}
     */
    public static List<Segment2D> exportSegments(Envelope2D _env) {
        return Arrays.asList(
                new ImmutableSegment2D(_env.getSouthWest(), _env.getSouthEast()),
                new ImmutableSegment2D(_env.getSouthWest(), _env.getNorthWest()),
                new ImmutableSegment2D(_env.getSouthEast(), _env.getNorthEast()),
                new ImmutableSegment2D(_env.getNorthWest(), _env.getNorthEast())
        );
    }

    /**
     * @param _env
     * @return
     */
    public static List<Point2D> exportCorners(Envelope2D _env) {
        return Arrays.asList(
                new ImmutablePoint2D(_env.getNorthEast()),
                new ImmutablePoint2D(_env.getSouthEast()),
                new ImmutablePoint2D(_env.getSouthWest()),
                new ImmutablePoint2D(_env.getNorthWest())
        );
    }

    /**
     * @param _env2
     * @return
     */
    public static Interval getXInterval(Envelope2D _env2) {
        return new Interval(_env2.getSouthWest().getX(), _env2.getNorthEast().getX());
    }

    /**
     * @param _env2
     * @return
     */
    public static Interval getYInterval(Envelope2D _env2) {
        return new Interval(_env2.getSouthWest().getY(), _env2.getNorthEast().getY());
    }

    /**
     * @param _env2
     * @return
     */
    public static Interval[] getXYInterval(Envelope2D _env2) {
        return new Interval[]{getXInterval(_env2), getYInterval(_env2)};
    }

    /**
     * @param _env3
     * @return
     */
    public static Interval getXInterval(Envelope3D _env3) {
        final List<Point3D> _bounds = _env3.getBounds();

        return new Interval(_bounds.get(0).getX(), _bounds.get(1).getX());
    }

    /**
     * @param _env3
     * @return
     */
    public static Interval getYInterval(Envelope3D _env3) {
        final List<Point3D> _bounds = _env3.getBounds();

        return new Interval(_bounds.get(0).getY(), _bounds.get(1).getY());
    }

    /**
     * @param _env3
     * @return
     */
    public static Interval getZInterval(Envelope3D _env3) {
        final List<Point3D> _bounds = _env3.getBounds();

        return new Interval(_bounds.get(0).getZ(), _bounds.get(1).getZ());
    }

    /**
     * @param _env3
     * @return
     */
    public static Interval[] getXYZInterval(Envelope3D _env3) {
        return new Interval[]{getXInterval(_env3), getYInterval(_env3), getZInterval(_env3)};
    }

    /**
     * @param _env2
     * @return
     */
    public static double getMinimumX(Envelope2D _env2) {
        return getXInterval(_env2).left;
    }

    /**
     * @param _env2
     * @return
     */
    public static double getMinimumX(Envelope3D _env3) {
        return getXInterval(_env3).left;
    }

    /**
     * @param _env2
     * @return
     */
    public static double getMinimumY(Envelope2D _env2) {
        return getYInterval(_env2).left;
    }

    /**
     * @param _env2
     * @return
     */
    public static double getMinimumY(Envelope3D _env3) {
        return getYInterval(_env3).left;
    }

    /**
     * @param _env2
     * @return
     */
    public static double getMaximumX(Envelope2D _env2) {
        return getXInterval(_env2).right;
    }

    /**
     * @param _env2
     * @return
     */
    public static double getMaximumX(Envelope3D _env3) {
        return getXInterval(_env3).right;
    }

    /**
     * @param _env2
     * @return
     */
    public static double getMaximumY(Envelope2D _env2) {
        return getYInterval(_env2).right;
    }

    /**
     * @param _env2
     * @return
     */
    public static double getMaximumY(Envelope3D _env3) {
        return getYInterval(_env3).right;
    }

    /**
     * @param _env2
     * @return
     */
    public static double getMinimumZ(Envelope3D _env3) {
        return getZInterval(_env3).left;
    }

    /**
     * @param _env2
     * @return
     */
    public static double getMaximumZ(Envelope3D _env3) {
        return getZInterval(_env3).right;
    }
}
