/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-algorithm project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.algorithm;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import terraml.algorithm.node.PQuadNode;
import static terraml.commons.Objects.isNull;
import static terraml.commons.Objects.nonNull;
import terraml.commons.tuple.NativeEntry;
import terraml.commons.unit.DirectionUnit;

/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
public class PQuadtree<Q> implements Collection<Map.Entry<Q, GeoPoint>>, Iterable<Map.Entry<Q, GeoPoint>> {

    public static final int DEFAULT_LIMIT = 256;

    // ben bu sınavdan kalmam.
    private HashMap<String, Map.Entry<Q, GeoPoint>> objects;
    private PQuadNode root = null;
    private final int limit;

    private DirectionCalculator directionCalculator;

    /**
     * @param dirCalc
     * @param quadrant
     * @param maxlimit
     */
    @SuppressWarnings("Convert2Lambda")
    public PQuadtree(DirectionCalculator dirCalc, Quadrant quadrant, int maxlimit) {

        if (isNull(dirCalc)) {
            this.directionCalculator = new DirectionCalculator() {
                @Override
                public DirectionUnit of(GeoPoint from, GeoPoint to) {
                    return fromAzimuth(from, to);
                }
            };
        } else {
            this.directionCalculator = dirCalc;
        }

        this.limit = maxlimit;
        this.objects = new HashMap<>(limit);

        this.root = new PQuadNode();
        this.root.setParent(null);
        this.root.setCoordinate(null);
        this.root.setQuadrant(quadrant);
        this.root.split();

    }

    /**
     * @param builder
     */
    public PQuadtree(PQuadtreeBuilder<Q> builder) {
        this(builder.getDirectionCalculator(), builder.getQuadrant(), builder.getLimit());
    }

    /**
     * @param dirCalc
     * @param quadrant
     */
    public PQuadtree(DirectionCalculator dirCalc, Quadrant quadrant) {
        this(dirCalc, quadrant, DEFAULT_LIMIT);
    }

    /**
     * @param dirCalc
     * @param p0
     * @param p1
     */
    public PQuadtree(DirectionCalculator dirCalc, GeoPoint p0, GeoPoint p1) {
        this(dirCalc, new Quadrant(p0, p1));
    }

    /**
     * @param p0
     * @param p1
     * @return
     */
    public Collection<Map.Entry<Q, GeoPoint>> query(GeoPoint p0, GeoPoint p1) {
        Quadrant quadrant = new Quadrant(p0, p1);

        PQuadNode quadNode = getRoot();
        Collection<Map.Entry<Q, GeoPoint>> collector = new ArrayList<>();

        return query(quadNode, collector, quadrant);
    }

    /**
     * @param src
     * @param collector
     * @param quadrant
     * @return
     */
    protected Collection<Map.Entry<Q, GeoPoint>> query(PQuadNode src,
                                                       Collection<Map.Entry<Q, GeoPoint>> collector,
                                                       Quadrant quadrant) {
        if (isNull(src)) {
            return collector;
        }

        // ! Consider better pre-check for here.
        // check whether if given Quadrant and tree's main quad interacts or not
        // It's good for preventing unecessary calculations.
        boolean intersectionExists = src.getQuadrant().intersects(quadrant);
        boolean containmentExists = src.getQuadrant().contains(quadrant);

        // if given rectangle and this quad is disjoint
        if (!intersectionExists && !containmentExists) {
            return collector;
        }

        // check whether is CELL or CELL_GROUP safely.
        boolean isCellGroup = isNull(src.getCoordinate());

        // if it's CELL_GROUP go thru every leg of quad.
        if (!isCellGroup) {
            GeoPoint coordinate = src.getCoordinate();
            String id = coordinate.getId();

            Map.Entry<Q, GeoPoint> entry = getObjects().get(id);

            if (quadrant.contains(coordinate)) {
                collector.add(entry);
            }
        }

        // güzel şeyler büyük acılarla gelir aldanma.
        collector = query(src.getNorthEastNode(), collector, quadrant);
        collector = query(src.getNorthWestNode(), collector, quadrant);
        collector = query(src.getSouthEastNode(), collector, quadrant);
        collector = query(src.getSouthWestNode(), collector, quadrant);

        return collector;
    }

    /**
     * @param quadNode
     * @param coordinate
     * @return
     */
    protected PQuadNode move(PQuadNode quadNode, GeoPoint coordinate) {
        DirectionUnit quadrant;

        PQuadNode reference = quadNode;
        while ( true ) {
            quadrant = directionCalculator.of(reference.getQuadrant().getCenter(), coordinate);

            if (quadrant.equals(DirectionUnit.NORTH_WEST)) {
                if (isNull(reference.getNorthWestNode())) {
                    break;
                }

                reference = reference.getNorthWestNode();
            } else if (quadrant.equals(DirectionUnit.NORTH_EAST)) {
                if (isNull(reference.getNorthEastNode())) {
                    break;
                }

                reference = reference.getNorthEastNode();
            } else if (quadrant.equals(DirectionUnit.SOUTH_EAST)) {
                if (isNull(reference.getSouthEastNode())) {
                    break;
                }

                reference = reference.getSouthEastNode();
            } else if (quadrant.equals(DirectionUnit.SOUTH_WEST)) {
                if (isNull(reference.getSouthWestNode())) {
                    break;
                }

                reference = reference.getSouthWestNode();
            }
        }

        // son sözümüz elbette hoşçakalın olmaz.
        return reference;
    }

    /**
     * @param reference
     * @param coordinate
     * @return
     */
    protected String add(PQuadNode reference, GeoPoint coordinate) {
        String id;

        reference = move(reference, coordinate);

        if (isNull(reference.getCoordinate())) {
            reference.setCoordinate(coordinate);
            id = coordinate.getId();
        } else {

            // umut her şeyden muaf.
            while ( nonNull(reference.getCoordinate()) ) {
                GeoPoint coord = reference.getCoordinate();
                reference.split();
                add(reference, coord);
            }

            reference.setCoordinate(coordinate);
            id = coordinate.getId();
        }

        return id;
    }

    /**
     * @param p0
     * @return
     */
    protected boolean inBounds(GeoPoint p0) {
        return Quadrant.contains(getRoot().getQuadrant(), p0);
    }

    /**
     * @param data
     * @param p0
     * @return
     */
    public String add(Q data, GeoPoint p0) {
        String id = null;

        if (inBounds(p0)) {
            GeoPoint coordinate = p0;

            id = add(getRoot(), coordinate);

            if (nonNull(id)) {
                getObjects().put(id, NativeEntry.of(data, p0));
            }
        }

        // mahalle ateşlenmiş basılmış evim.
        return id;
    }

    @Override
    public boolean add(Map.Entry<Q, GeoPoint> entry) {
        return add(entry.getKey(), entry.getValue()) != null;
    }

    @Override
    public boolean addAll(Collection<? extends Map.Entry<Q, GeoPoint>> c) {
        for ( Map.Entry<Q, GeoPoint> each : c ) {
            if (!add(each)) {
                return false;
            }
        }

        return true;
    }

    /**
     * @param coordinate
     * @return
     */
    protected boolean contains(GeoPoint coordinate) {
        DirectionUnit quadrant;
        PQuadNode reference = getRoot();

        boolean found = false;
        while ( true ) {
            if (nonNull(reference.getCoordinate()) && reference.getCoordinate().equals(coordinate)) {
                found = true;
                break;
            }

            // dönmedim geri, çıkmıyor üstümden şu geçmişin kiri.
            quadrant = directionCalculator.of(reference.getQuadrant().getCenter(), coordinate);
            if (quadrant.equals(DirectionUnit.NORTH_WEST)) {
                if (isNull(reference.getNorthWestNode())) {
                    break;
                }

                reference = reference.getNorthWestNode();
            } else if (quadrant.equals(DirectionUnit.NORTH_EAST)) {
                if (isNull(reference.getNorthEastNode())) {
                    break;
                }

                reference = reference.getNorthEastNode();
            } else if (quadrant.equals(DirectionUnit.SOUTH_EAST)) {
                if (isNull(reference.getSouthEastNode())) {
                    break;
                }

                reference = reference.getSouthEastNode();
            } else if (quadrant.equals(DirectionUnit.SOUTH_WEST)) {
                if (isNull(reference.getSouthWestNode())) {
                    break;
                }

                reference = reference.getSouthWestNode();
            }
        }

        return found;
    }

    /**
     * @param data
     * @param geoPoint
     * @return
     */
    public boolean contains(Q data, GeoPoint geoPoint) {
        if (isNull(geoPoint.getId())) {
            return false;
        }

        Map.Entry<Q, GeoPoint> entry = getObjects().get(geoPoint.getId());

        boolean keyMatch = entry.getKey().equals(data);
        boolean valMatch = entry.getValue().equals(geoPoint);

        return keyMatch && valMatch;
    }

    /**
     * @param entry
     * @return
     */
    public boolean contains(Map.Entry<Q, GeoPoint> entry) {
        return contains(entry.getKey(), entry.getValue());
    }

    /**
     * @param object
     * @return
     */
    @Override
    public boolean contains(Object object) {
        Map.Entry<Q, GeoPoint> entry = (Map.Entry<Q, GeoPoint>) object;

        return contains(entry);
    }

    @Override
    public boolean containsAll(Collection<?> c) {
        for ( Object each : c ) {
            if (!contains(each)) {
                return false;
            }
        }

        return true;
    }

    /**
     * @param quadNode
     * @param collector
     * @return
     */
    protected List<Quadrant> boundries(PQuadNode quadNode, List<Quadrant> collector) {
        if (quadNode == null) {
            return collector;
        }

        if (quadNode.getQuadrant() != null) {
            Quadrant temp = quadNode.getQuadrant();

            // bura kimlerin alanı,
            // bunlar kimlerin yalanı,
            // bugün kirlenir sokağım.
            collector.add(temp);

        }

        collector = boundries(quadNode.getNorthEastNode(), collector);
        collector = boundries(quadNode.getNorthWestNode(), collector);
        collector = boundries(quadNode.getSouthEastNode(), collector);
        collector = boundries(quadNode.getSouthWestNode(), collector);

        return collector;
    }

    /**
     * @return
     */
    public Enumeration<Quadrant> boundries() {
        PQuadNode reference = getRoot();

        List<Quadrant> list = boundries(reference, new ArrayList<>());
        final int length = list.size();

        return new Enumeration<Quadrant>() {
            int index = 0;

            @Override
            public boolean hasMoreElements() {
                return index < length;
            }

            @Override
            public Quadrant nextElement() {
                if (hasMoreElements()) {
                    return list.get(index++);
                }

                throw new NoSuchElementException("NoSuchElementException");
            }
        };
    }

    /**
     * @param quadNode
     * @param collector
     * @return
     */
    protected Collection<GeoPoint> distinct(PQuadNode quadNode, Collection<GeoPoint> collector) {
        if (isNull(quadNode)) {
            return collector;
        } else if (nonNull(quadNode.getCoordinate())) {
            collector.add(quadNode.getCoordinate());
        }

        collector = distinct(quadNode.getNorthEastNode(), collector);
        collector = distinct(quadNode.getNorthWestNode(), collector);
        collector = distinct(quadNode.getSouthEastNode(), collector);
        collector = distinct(quadNode.getSouthWestNode(), collector);

        return collector;
    }

    @Override
    public Iterator<Map.Entry<Q, GeoPoint>> iterator() {
        return getObjects().values().iterator();
    }

    /**
     * @return
     */
    public Collection<Map.Entry<Q, GeoPoint>> collection() {
        return getObjects().values();
    }

    @Override
    public Object[] toArray() {
        final PQuadNode ref = getRoot();

        return distinct(ref, new ArrayList<>()).toArray();
    }

    @Override
    public <T> T[] toArray(T[] a) {
        final PQuadNode ref = getRoot();

        return distinct(ref, new ArrayList<>()).toArray(a);
    }

    @Override
    public int size() {
        if (isNull(getObjects())) {
            return 0;
        }

        return getObjects().keySet().size();
    }

    @Override
    public void clear() {
        setObjects(null);
        setRoot(null);
    }

    /**
     * @return
     */
    @Override
    public boolean isEmpty() {
        return isNull(getRoot());
    }

    @Override
    public boolean remove(Object o) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    @Override
    public boolean removeAll(Collection<?> c) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    @Override
    public boolean retainAll(Collection<?> c) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     * @return
     */
    public PQuadNode getRoot() {
        return root;
    }

    /**
     * @param root
     */
    public void setRoot(PQuadNode root) {
        this.root = root;
    }

    /**
     * @return
     */
    public HashMap<String, Map.Entry<Q, GeoPoint>> getObjects() {
        return objects;
    }

    /**
     * @param objects
     */
    public void setObjects(HashMap<String, Map.Entry<Q, GeoPoint>> objects) {
        this.objects = objects;
    }

    /**
     * @return
     */
    public int getLimit() {
        return limit;
    }
}
