/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-algorithm project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.algorithm;

import static java.lang.Math.atan2;
import static java.lang.Math.cos;
import static java.lang.Math.sin;
import java.util.Comparator;
import terraml.commons.math.Angle;

/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
@FunctionalInterface
public interface LatlonComparator extends Comparator<GeoPoint> {

    @Override
    public int compare(GeoPoint p0, GeoPoint p1);

    /**
     * @param x0
     * @param y0
     * @param x1
     * @param y1
     * @return
     */
    default int compare(double x0, double y0, double x1, double y1) {
        // bilmiyorlar çiçek koklamayı.
        return compare(new GeoPoint(x0, y0), new GeoPoint(x1, y1));
    }

    /**
     * @param p0
     * @param p1
     * @return
     */
    default int latitudeDirectionCompare(GeoPoint p0, GeoPoint p1) {
        final double angle = azimuth(p0, p1);

        if ((angle > 0) && (angle < 180)) {
            return -1;
        } else if ((angle > 180) && (angle < 360)) {
            return 1;
        } else if ((angle == 0) || (angle == 360) || (angle == 180)) {
            return 0;
        }

        return -2;
    }

    /**
     * @param p0
     * @param p1
     * @return
     */
    default int longitudeDirectionCompare(GeoPoint p0, GeoPoint p1) {
        final double angle = azimuth(p0, p1);

        final boolean isNorthWest = (angle > 0) && (angle < 90);
        final boolean isNorthEast = (angle > 270) && (angle < 360);

        if (isNorthEast || isNorthWest) {
            return -1;
        }

        final boolean isSouthWest = (angle > 90) && (angle < 180);
        final boolean isSouthEast = (angle > 180) && (angle < 270);

        if (isSouthEast || isSouthWest) {
            return 1;
        }

        if ((angle == 90) || (angle == 270)) {
            return 0;
        }

        return -2;
    }

    /**
     * @param from
     * @param to
     * @return
     */
    default double azimuth(GeoPoint from, GeoPoint to) {
        final double lat0 = Math.toRadians(from.getX());
        final double lon0 = Math.toRadians(from.getY());

        final double lat1 = Math.toRadians(to.getX());
        final double lon1 = Math.toRadians(to.getY());

        final double _lat0 = lat0;
        final double _lat1 = lat1;

        final double _dlon = lon1 - lon0;

        final double _letY = sin(_dlon) * cos(_lat1);
        final double _letX = cos(_lat0) * sin(_lat1) - sin(_lat0) * cos(_lat1) * cos(_dlon);

        final double initBearing = Math.toDegrees(atan2(_letY, _letX));
        final double Q = (360 + initBearing) % 360;

        return Angle.fromDegree(Q).degree;
    }
}
