/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-algorithm project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.algorithm;

import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.function.Predicate;
import terraml.algorithm.node.HashtableNode;

/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version PUBLIC-1.0
 */
public class Hashtable<K, V> {

    private int node_count;
    private int total_count;

    // sen ah çekip yerinde say, ben [] ikiye katlarım.
    private HashtableNode<K, V>[] hashtable;
    private static final int INITIAL_SIZE = 256;

    private static final double load_factor = 0.75;
    private static final double increase_percent = 0.5;

    /**
     *
     */
    public Hashtable() {
        this.initTable();
    }

    // Business logic.
    private boolean putKV(K key, V value) {
        this.capacity_control();
        for ( int r = 0; r < INITIAL_SIZE; r++ ) {
            int idx = this.hash_decoder(key, r, INITIAL_SIZE);
            if (hashtable[idx] == null) {
                hashtable[idx] = new HashtableNode<>(key, value);
                ++node_count;
                ++total_count;

                return true;
            } else if ((hashtable[idx] != null) && (hashtable[idx].getKey().equals(key))) {
                hashtable[idx].add(value);
                ++total_count;

                return true;
            }
        }

        return false;
    }

    // Business logic.
    private <E> int putAll(K key, E values) {
        int counter = 0;
        if (values instanceof Object[]) {
            for ( V each : ((V[]) values) ) {
                counter += this.putKV(key, each) ? 1 : 0;
            }
        } else if (values instanceof List) {
            for ( V each : ((List<V>) values) ) {
                counter += this.putKV(key, each) ? 1 : 0;
            }
        }

        return counter;
    }

    /**
     * Method puts given entry to table.
     *
     * @param key   key
     * @param value value
     * @return true if given entry puts successfully. Otherwise it returns
     *         false.
     */
    public boolean put(K key, V value) {
        return this.putKV(key, value);
    }

    /**
     * Method puts given entry to table.
     *
     * @param key    key
     * @param values array of values
     * @return true if given entry puts successfully. Otherwise it returns
     *         false.
     */
    public int put(K key, V[] values) {
        return this.putAll(key, values);
    }

    /**
     * Method puts given entry to table.
     *
     * @param key    key
     * @param values List of values
     * @return true if given entry puts successfully. Otherwise it returns
     *         false.
     */
    public int put(K key, List<V> values) {
        return this.putAll(key, values);
    }

    // Business logic.
    private boolean containsK(K key) {
        for ( HashtableNode<K, V> each : this.hashtable ) {
            if ((each != null) && (each.equals(key))) {
                return true;
            }
        }

        return false;
    }

    // Business logic.
    private boolean containsKV(K key, V value) {
        for ( HashtableNode<K, V> each : this.hashtable ) {
            if ((each != null) && (each.equals(key)) && (each.getValues().contains(value))) {
                return true;
            }
        }

        return false;
    }

    // Business logic.
    private boolean containsKV(K key, List<V> values) {
        boolean safe_check = false;
        for ( HashtableNode<K, V> each : this.hashtable ) {
            if ((each != null) && each.equals(key)) {
                safe_check = true;
                for ( V rsz_each : values ) {
                    if (!each.getValues().contains(rsz_each)) {
                        return false;
                    }
                }
            }
        }

        return safe_check;
    }

    /**
     * Method checks hashtable contains key or not.
     *
     * @param key key to search.
     * @return true if key table contains given key. Otherwise it returns false.
     */
    public boolean contains(K key) {
        return this.containsK(key);
    }

    /**
     * Method checks hashtable contains key and value or not.
     *
     * @param key   key to search.
     * @param value value to search
     * @return true if key table contains given key and value. Otherwise it
     *         returns false.
     */
    public boolean contains(K key, V value) {
        return this.containsKV(key, value);
    }

    /**
     * Method checks hashtable contains key and List of values or not.
     *
     * @param key    key to search.
     * @param values List of value to search
     * @return true if key table contains given key and List of values.
     *         Otherwise it returns false.
     */
    public boolean contains(K key, List<V> values) {
        return this.containsKV(key, values);
    }

    // Business logic.
    private List<V> getV(K key) {
        for ( int r = 0; r < INITIAL_SIZE; r++ ) {
            int idx = this.hash_decoder(key, r, INITIAL_SIZE);

            if ((hashtable[idx] != null) && hashtable[idx].getKey().equals(key)) {
                return hashtable[idx].getValues();
            }
        }

        return null;
    }

    /**
     * Method returns given key's values.
     *
     * @param key key
     * @return values if key hashtable contains given key. Otherwise it returns
     *         null
     */
    public List<V> get(K key) {
        return this.getV(key);
    }

    // Business logic.
    private List<V> getIfV(K key, Predicate filter) {
        List<V> newValues = new ArrayList<>();

        for ( int r = 0; r < INITIAL_SIZE; r++ ) {
            int idx = this.hash_decoder(key, r, INITIAL_SIZE);

            if ((hashtable[idx] != null) && hashtable[idx].getKey().equals(key)) {
                hashtable[idx].getValues().forEach(each -> {
                    if (filter.test(each)) {
                        newValues.add(each);
                    }
                });

                return newValues;
            }
        }

        return null;
    }

    /**
     * Method returns List that given keys values provides given filter.
     *
     * @param key    key
     * @param filter Predicate
     * @return given key's List of values if given filter provides.
     */
    public List<V> getIf(K key, Predicate filter) {
        return this.getIfV(key, filter);
    }

    // Business logic.
    private boolean removeK(K key) {
        for ( int r = 0; r < INITIAL_SIZE; r++ ) {
            int idx = this.hash_decoder(key, r, INITIAL_SIZE);

            if ((hashtable[idx] != null) && hashtable[idx].getKey().equals(key)) {
                total_count -= hashtable[idx].getValues().size();
                --node_count;
                hashtable[idx] = null;

                return true;
            }
        }

        return false;
    }

    /**
     * Method removes given key and it's values from hashtable.
     *
     * @param key key
     * @return true if given key and it's values removed. Otherwise it returns
     *         false.
     */
    public boolean remove(K key) {
        return this.removeK(key);
    }

    // Business logic.
    private boolean removeKV(K key, V value) {
        for ( int r = 0; r < INITIAL_SIZE; r++ ) {
            int idx = this.hash_decoder(key, r, INITIAL_SIZE);

            if ((hashtable[idx] != null) && hashtable[idx].getKey().equals(key)) {
                if (hashtable[idx].getValues().remove(value)) {
                    --total_count;
                }

                return true;
            }
        }

        return false;
    }

    /**
     *
     * @param key
     * @param value
     * @return
     */
    public boolean remove(K key, V value) {
        return this.removeKV(key, value);
    }

    // Business logic.
    private void initTable() {
        this.hashtable = (HashtableNode<K, V>[]) Array.newInstance(HashtableNode.class, INITIAL_SIZE);
        this.total_count = 0;
        this.node_count = 0;
    }

    /**
     *
     */
    public void clear() {
        this.initTable();
    }

    // Business logic.
    private boolean replaceK(K key, V oldValue, V newValue) {
        for ( int r = 0; r < INITIAL_SIZE; r++ ) {
            int idx = this.hash_decoder(key, r, INITIAL_SIZE);

            if ((hashtable[idx] != null) && hashtable[idx].getKey().equals(key)) {
                if (hashtable[idx].getValues().remove(oldValue)) {
                    return hashtable[idx].getValues().add(newValue);
                }
            }

        }

        return false;
    }

    /**
     *
     * @param key
     * @param oldValue
     * @param newValue
     * @return
     */
    public boolean replace(K key, V oldValue, V newValue) {
        return this.replaceK(key, oldValue, newValue);
    }

    // Business logic.
    private List<V> replaceKV(K key, List<V> newValues) {
        List<V> oldValues = new ArrayList<>();
        for ( int r = 0; r < INITIAL_SIZE; r++ ) {
            int idx = this.hash_decoder(key, r, INITIAL_SIZE);
            if ((hashtable[idx] != null) && hashtable[idx].getKey().equals(key)) {

                if (hashtable[idx].getValues().size() < newValues.size()) {
                    this.capacity_control(newValues.size() - hashtable[idx].getValues().size());
                }

                oldValues.addAll(hashtable[idx].getValues());
                int rmv_size = oldValues.size();
                hashtable[idx].getValues().clear();
                hashtable[idx].getValues().addAll(newValues);
                int add_size = hashtable[idx].getValues().size();

                total_count -= rmv_size;
                total_count += add_size;

                return oldValues;
            }

        }

        return null;
    }

    /**
     *
     * @param key
     * @param newValues
     * @return
     */
    public List<V> replace(K key, List<V> newValues) {
        return this.replaceKV(key, newValues);
    }

    // Business logic.
    private void capacity_control() {
        if ((INITIAL_SIZE * load_factor) < this.node_count) {
            int increase = (int) (this.node_count + (this.node_count * increase_percent));
            this.hashtable = Arrays.copyOf(this.hashtable, increase);
        }
    }

    // Business logic.
    private void capacity_control(int len) {
        int line = (int) (INITIAL_SIZE * load_factor);
        int limit = this.node_count + len;

        if (line < limit) {
            int increase = (int) (limit + (limit * increase_percent));
            this.hashtable = Arrays.copyOf(this.hashtable, increase);
        }
    }

    /**
     * Method decodes the index by given argument
     *
     * @param key   arg
     * @param point arg
     * @param size  arg
     * @return decoded index
     */
    private int hash_decoder(K key, int point, int size) {
        return ((hash_function(key)
                 + ((int) (Math.pow(point, 2) + point) >> 2)) % size);
    }

    /**
     * @param key to encode
     * @return the hash value
     */
    private int hash_function(K key) {
        String toHash = key.toString();
        int hashValue = 0;

        for ( int pos = 0; pos < toHash.length(); ++pos ) {
            hashValue = (hashValue << 4) + toHash.charAt(pos);
            int highBits = hashValue & 0xF0000000;

            if (highBits != 0) {
                hashValue ^= highBits >> 24;
            }

            hashValue &= ~highBits;
        }

        return hashValue;
    }

    /**
     *
     * @return
     */
    public int size() {
        return node_count;
    }

    /**
     *
     * @return
     */
    public int length() {
        return total_count;
    }
}
