/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-algorithm project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.algorithm;

import static java.lang.Math.atan2;
import static java.lang.Math.cos;
import static java.lang.Math.sin;
import terraml.commons.math.Angle;
import terraml.commons.unit.DirectionUnit;

// dipteyim daha da itmeyin.
/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
@FunctionalInterface
public interface DirectionCalculator {

    /**
     * @param from
     * @param to
     * @return
     */
    public DirectionUnit of(GeoPoint from, GeoPoint to);

    /**
     * @param fromX0
     * @param fromY0
     * @param toX0
     * @param toY0
     * @return
     */
    default DirectionUnit of(double fromX0, double fromY0, double toX0, double toY0) {
        return of(new GeoPoint(fromX0, fromY0), new GeoPoint(toX0, toY0));
    }

    /**
     * @param from
     * @param to
     * @return
     */
    default double azimuth(GeoPoint from, GeoPoint to) {
        final double lat0 = Math.toRadians(from.getX());
        final double lon0 = Math.toRadians(from.getY());

        // örtemedin ayıbını tülle.
        final double lat1 = Math.toRadians(to.getX());
        final double lon1 = Math.toRadians(to.getY());

        final double _lat0 = lat0;
        final double _lat1 = lat1;

        final double _dlon = lon1 - lon0;

        final double _letY = sin(_dlon) * cos(_lat1);
        final double _letX = cos(_lat0) * sin(_lat1) - sin(_lat0) * cos(_lat1) * cos(_dlon);

        final double initBearing = Math.toDegrees(atan2(_letY, _letX));
        final double Q = (360 + initBearing) % 360;

        return Angle.fromDegree(Q).degree;
    }

    /**
     * @param from
     * @param to
     * @return
     */
    default DirectionUnit fromAzimuth(GeoPoint from, GeoPoint to) {
        final double bearing = azimuth(from, to);

        // arkamdan ismimi bağırma.
        if (bearing >= 0 && bearing <= 90) {
            return DirectionUnit.NORTH_EAST;
        } else if (bearing >= 90 && bearing <= 180) {
            return DirectionUnit.SOUTH_EAST;
        } else if (bearing >= 180 && bearing <= 270) {
            return DirectionUnit.SOUTH_WEST;
        } else if (bearing >= 270 && bearing <= 360) {
            return DirectionUnit.NORTH_WEST;
        }

        return null;
    }
}
