/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-algorithm project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.algorithm;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.SortedSet;
import java.util.Spliterator;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.stream.Stream;
import terraml.algorithm.iterator.AVLOrderedTraversal;
import terraml.algorithm.node.AVLNode;
import static terraml.commons.Objects.isNull;
import static terraml.commons.Objects.nonNull;

/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
public class AVLSearch<Q> implements BiTreeSet<Q> {

    // diskografi ile insanlık mı ölçülürmüş ?
    private final Comparator<Q> comparator;
    private AVLNode<Q> root;

    /**
     * @param comparator
     */
    public AVLSearch(Comparator<Q> comparator) {
        this.comparator = comparator;
    }

    /**
     * @param comparator
     * @param root
     */
    public AVLSearch(Comparator<Q> comparator, AVLNode<Q> root) {
        this(comparator);
        this.root = root;
    }

    /**
     * @param comparator
     * @param data
     */
    public AVLSearch(Comparator<Q> comparator, Q data) {
        this(comparator, new AVLNode<>(comparator, data));
    }

    @Override
    public boolean add(Q data) {
        if (isNull(getRoot())) {

            setRoot(new AVLNode<>(getComparator(), null, data));
            return true;

        }

        AVLNode<Q> reference = getRoot();
        while ( true ) {

            if (eq(data, reference)) {
                return reference.addDuplicate(data);
            }

            AVLNode<Q> parent = reference;

            boolean lefty = st(data, reference);

            // aklımın odaları karışık.
            reference = lefty ? reference.getLeft() : reference.getRight();

            if (isNull(reference)) {

                if (lefty) {
                    parent.setLeft(new AVLNode<>(getComparator(), parent, data));
                } else {
                    parent.setRight(new AVLNode<>(getComparator(), parent, data));
                }

                rebalance(parent);
                break;

            }

        }

        return true;
    }

    @Override
    public boolean addAll(Collection<? extends Q> collection) {

        for ( Q each : collection ) {
            if (!add(each)) {
                return false;
            }
        }

        return true;

    }

    /**
     * @param src
     * @param data
     * @return
     */
    protected boolean contains(AVLNode<Q> src, Q data) {
        if (isNull(src)) {
            return false;
        }

        if (eq(data, src)) {

            if (data.equals(src.getData())) {
                return true;
            }

            if (src.hasDuplicates()) {

                for ( Q each : src.getDuplicates() ) {

                    if (data.equals(each)) {
                        return true;
                    }

                }

            }

        }

        if (st(data, src)) {
            return contains(src.getLeft(), data);
        } else if (gt(data, src)) {
            return contains(src.getRight(), data);
        }

        return false;
    }

    @Override
    public boolean contains(Object object) {
        final AVLNode<Q> ref = getRoot();

        return contains(ref, (Q) object);
    }

    @Override
    public boolean containsAll(Collection<?> collection) {
        for ( Object each : collection ) {
            if (!contains(each)) {
                return false;
            }
        }

        return true;
    }

    /**
     * @param src
     * @param data
     * @return
     */
    protected boolean containsFamily(AVLNode<Q> src, Q data) {
        if (isNull(src)) {
            return false;
        }

        if (eq(data, src)) {

            return true;

        }

        if (st(data, src)) {
            return contains(src.getLeft(), data);
        } else if (gt(data, src)) {
            return contains(src.getRight(), data);
        }

        return false;
    }

    @Override
    public boolean containsFamily(Q object) {
        final AVLNode<Q> ref = getRoot();

        return containsFamily(ref, object);
    }

    @Override
    public boolean containsAllFamily(Collection<Q> collection) {
        for ( Q each : collection ) {
            if (!containsFamily(each)) {
                return false;
            }
        }

        return true;
    }

    /**
     * @param data
     * @return
     */
    @Override
    public void removeFamily(Q object) {
        if (isNull(getRoot())) {
            return;
        }

        @SuppressWarnings("unchecked")

        AVLNode<Q> child = getRoot();
        while ( nonNull(child) ) {
            AVLNode<Q> node = child;

            child = gt(object, node.getData()) ? node.getRight() : node.getLeft();

            if (eq(object, node)) {

                _removeFamily(node);

            }
        }

    }

    @Override
    public void removeAllFamily(Collection<? extends Q> collection) {

        for ( Q each : collection ) {
            removeFamily(each);
        }

    }

    @Override
    public boolean removeIf(Predicate<? super Q> filter) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    @Override
    public boolean remove(Object object) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    @Override
    public boolean removeAll(Collection<?> collection) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     * @param node
     */
    protected void _removeFamily(AVLNode<Q> node) {
        if (lf(node)) {

            if (isNull(node.getParent())) {

                if (getRoot().equals(node)) {
                    setRoot(null);
                }

            } else {

                AVLNode<Q> parent = node.getParent();
                if (eq(parent.getLeft(), node)) {
                    parent.setLeft(null);
                } else {
                    parent.setRight(null);
                }

                rebalance(parent);

            }

            return;

        }

        if (!lf(node)) {
            AVLNode<Q> child = node.getLeft();
            while ( !re(child) ) {

                child = child.getRight();

            }

            node.setData(child.getData());
            _removeFamily(child);

        } else {
            AVLNode<Q> child = node.getRight();
            while ( !le(child) ) {

                child = child.getLeft();

            }

            node.setData(child.getData());
            _removeFamily(child);

        }
    }

    /**
     * @param node
     */
    protected void rebalance(AVLNode<Q> node) {
        setBalance(Arrays.asList(node));

        if (node.getBalance() == -2) {

            if (height(node.getLeft().getLeft()) >= height(node.getLeft().getRight())) {
                node = rotateRight(node);
            } else {
                node = rotateLeftThenRight(node);
            }

        } else if (node.getBalance() == 2) {

            if (height(node.getRight().getRight()) >= height(node.getRight().getLeft())) {
                node = rotateLeft(node);
            } else {
                node = rotateRightThenLeft(node);
            }
        }

        if (nonNull(node.getParent())) {
            rebalance(node.getParent());
        } else {
            setRoot(node);
        }
    }

    /**
     * @param node
     * @return
     */
    protected AVLNode<Q> rotateLeft(AVLNode<Q> node) {

        AVLNode<Q> reference = node.getRight();
        reference.setParent(node.getParent());

        node.setRight(reference.getLeft());

        if (nonNull(node.getRight())) {
            node.getRight().setParent(node);
        }

        reference.setLeft(node);
        node.setParent(reference);

        if (nonNull(reference.getParent())) {

            if (isNull(reference.getParent().getRight())) {
                reference.getParent().setLeft(reference);
            } else {
                if (reference.getParent().getRight().equals(node)) {
                    reference.getParent().setRight(reference);
                } else {
                    reference.getParent().setLeft(reference);
                }
            }

        }

        setBalance(Arrays.asList(node, reference));

        return reference;
    }

    /**
     * @param node
     * @return
     */
    protected AVLNode<Q> rotateRight(AVLNode<Q> node) {

        AVLNode<Q> reference = node.getLeft();
        reference.setParent(node.getParent());

        node.setLeft(reference.getRight());

        if (nonNull(node.getLeft())) {
            node.getLeft().setParent(node);
        }

        reference.setRight(node);
        node.setParent(reference);

        if (nonNull(reference.getParent())) {

            if (isNull(reference.getParent().getRight())) {
                reference.getParent().setLeft(reference);
            } else {
                if (reference.getParent().getRight().equals(node)) {
                    reference.getParent().setRight(reference);
                } else {
                    reference.getParent().setLeft(reference);
                }
            }

        }

        setBalance(Arrays.asList(node, reference));

        // odalarda ışıksızım.
        return reference;
    }

    /**
     * @param node
     * @return
     */
    protected AVLNode<Q> rotateLeftThenRight(AVLNode<Q> node) {
        node.setLeft(rotateLeft(node.getLeft()));

        return rotateRight(node);
    }

    /**
     * @param node
     * @return
     */
    protected AVLNode<Q> rotateRightThenLeft(AVLNode<Q> node) {
        node.setRight(rotateRight(node.getRight()));

        return rotateLeft(node);
    }

    /**
     * @param src
     * @return
     */
    protected Q minimum(AVLNode<Q> src) {
        if (le(src)) {
            return src.getData();
        }

        return minimum(src.getLeft());
    }

    /**
     * @param src source
     * @return maximum comparable object.
     */
    protected Q maximum(AVLNode<Q> src) {
        if (re(src)) {
            return src.getData();
        }

        return maximum(src.getRight());
    }

    @Override
    public Q last() {
        return maximum(getRoot());
    }

    @Override
    public Q first() {
        return minimum(getRoot());
    }

    /**
     * @param src source
     * @return node that hold minimum comparable objects.
     */
    protected AVLNode<Q> minimumFamily(AVLNode<Q> src) {
        if (le(src)) {
            return src;
        }

        return minimumFamily(src.getLeft());
    }

    /**
     * @param src source
     * @return node that hold maximum comparable objects.
     */
    protected AVLNode<Q> maximumFamily(AVLNode<Q> src) {
        if (re(src)) {
            return src;
        }

        return maximumFamily(src.getRight());
    }

    @Override
    public Iterator<Q> lastFamily() {
        // find last
        AVLNode<Q> bi = minimumFamily(getRoot());

        List<Q> list = new ArrayList<>();

        // add primary data
        list.add(bi.getData());

        // if there are duplicates
        if (bi.hasDuplicates()) {

            // add each one of them to the list.
            for ( Q each : bi.getDuplicates() ) {
                list.add(each);
            }

        }

        return list.iterator();
    }

    @Override
    public Iterator<Q> firstFamily() {
        // find first
        AVLNode<Q> bi = maximumFamily(getRoot());

        List<Q> list = new ArrayList<>();

        // add primary data
        list.add(bi.getData());

        // if there are duplicates
        if (bi.hasDuplicates()) {

            // add each one of them to the list.
            for ( Q each : bi.getDuplicates() ) {
                list.add(each);
            }

        }

        return list.iterator();
    }

    /**
     * @param src
     * @param item
     * @return
     */
    protected AVLNode<Q> tailSet(AVLNode<Q> src, Q item) {
        if (src == null) {
            return null;
        }

        while ( (eq(item, src) || gt(item, src)) && src != null ) {
            src = tailSet(src.getRight(), item);
        }

        if (st(item, src)) {
            src.setLeft(tailSet(src.getLeft(), item));
        }

        return src;
    }

    @Override
    public SortedSet<Q> tailSet(Q fromElement) {
        AVLNode<Q> bi = getRoot();
        bi = tailSet(bi, fromElement);

        return new AVLSearch<>(comparator, bi);
    }

    /**
     * @param src
     * @param item
     * @return
     */
    protected AVLNode<Q> headSet(AVLNode<Q> src, Q item) {
        if (src == null) {
            return null;
        }

        while ( (eq(item, src) || st(item, src)) && src != null ) {
            src = headSet(src.getLeft(), item);
        }

        if (gt(item, src)) {
            src.setRight(headSet(src.getRight(), item));
        }

        return src;
    }

    @Override
    public SortedSet<Q> headSet(Q toElement) {
        AVLNode<Q> bi = getRoot();
        bi = headSet(bi, toElement);

        return new AVLSearch<>(comparator, bi);
    }

    /**
     * @param src
     * @param from
     * @param to
     * @return
     */
    protected AVLNode<Q> subSet(AVLNode<Q> src, Q from, Q to) {
        if (src == null) {
            return null;
        }

        if (st(to, src)) {
            return subSet(src.getLeft(), from, to);
        } else if (gt(from, src)) {
            return subSet(src.getRight(), from, to);
        }

        src.setLeft(subSet(src.getLeft(), from, to));
        src.setRight(subSet(src.getRight(), from, to));

        return src;
    }

    @Override
    public SortedSet<Q> subSet(Q fromElement, Q toElement) {
        AVLNode<Q> bi = getRoot();
        bi = subSet(bi, fromElement, toElement);

        return new AVLSearch<>(comparator, bi);
    }

    /**
     * @param src
     * @param list
     * @return
     */
    protected List<Q> collect(AVLNode<Q> src, List<Q> list) {
        if (src == null) {
            return list;
        }

        // add head node data
        list.add(src.getData());

        // if node has duplicates
        if (src.hasDuplicates()) {

            // add all
            for ( Q each : src.getDuplicates() ) {
                list.add(each);
            }

        }

        // branch left
        list = collect(src.getLeft(), list);

        // branch right
        list = collect(src.getRight(), list);

        return list;
    }

    /**
     * @param src
     * @return
     */
    protected int size(AVLNode<Q> src) {
        if (src == null) {
            return 0;
        }

        int count = 1;
        if (src.hasDuplicates()) {
            count += src.getDuplicates().size();
        }

        return count + size(src.getLeft()) + size(src.getRight());
    }

    @Override
    public int size() {
        return size(getRoot());
    }

    @Override
    public Spliterator<Q> spliterator() {
        return collect(getRoot(), new ArrayList<>()).spliterator();
    }

    @Override
    public Comparator<? super Q> comparator() {
        return this.comparator;
    }

    @Override
    public void clear() {
        setRoot(null);
    }

    @Override
    public boolean retainAll(Collection<?> collection) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    @Override
    public <T> T[] toArray(T[] a) {
        return collect(getRoot(), new ArrayList<>()).toArray(a);
    }

    @Override
    public Object[] toArray() {
        return collect(getRoot(), new ArrayList<>()).toArray();
    }

    @Override
    public Iterator<Q> iterator() {
        return new AVLOrderedTraversal<>(getRoot());
    }

    @Override
    public boolean isEmpty() {
        return isNull(getRoot());
    }

    @Override
    public Stream<Q> parallelStream() {
        return collect(getRoot(), new ArrayList<>()).parallelStream();
    }

    @Override
    public Stream<Q> stream() {
        return collect(getRoot(), new ArrayList<>()).stream();
    }

    @Override
    public void forEach(Consumer<? super Q> action) {
        collect(getRoot(), new ArrayList<>()).forEach(action);
    }

    /**
     * @param nodes
     */
    protected void setBalance(Collection<AVLNode<Q>> nodes) {
        // acaba neden gerilim hat safhalarda ?
        for ( AVLNode<Q> each : nodes ) {
            reheight(each);

            int balance = height(each.getRight()) - height(each.getLeft());

            each.setBalance(balance);
        }
    }

    /**
     * @param data comparable object
     * @param src  source
     * @return true if comparison result equals to 0. Otherwise it returns false. Also
     *         if any given arguments is null then it returns false.
     */
    protected boolean eq(AVLNode<Q> src, AVLNode<Q> tar) {
        return src != null && tar != null && comparator.compare(src.getData(), tar.getData()) == 0;
    }

    /**
     * @param data comparable object
     * @param src  source
     * @return true if comparison result equals to 0. Otherwise it returns false. Also
     *         if any given arguments is null then it returns false.
     */
    protected boolean eq(Q data, AVLNode<Q> src) {
        return data != null && src != null && comparator.compare(data, src.getData()) == 0;
    }

    /**
     * @param data comparable object
     * @param src  source
     * @return true if comparison result equals to 0. Otherwise it returns false. Also
     *         if any given arguments is null then it returns false.
     */
    protected boolean eq(Q data0, Q data1) {
        return data0 != null && data1 != null && comparator.compare(data0, data1) == 0;
    }

    /**
     * @param data comparable object
     * @param src  source
     * @return true if comparison result equals to 0. Otherwise it returns false. Also
     *         if any given arguments is null then it returns false.
     */
    protected boolean st(AVLNode<Q> src, AVLNode<Q> tar) {
        return src != null && tar != null && comparator.compare(src.getData(), tar.getData()) < 0;
    }

    /**
     * @param data comparable object
     * @param src  source
     * @return true if comparison result smaller than 0. Otherwise it returns false. Also
     *         if any given arguments is null then it returns false.
     */
    protected boolean st(Q data, AVLNode<Q> src) {
        return data != null && src != null && comparator.compare(data, src.getData()) < 0;
    }

    /**
     * @param data comparable object
     * @param src  source
     * @return true if comparison result equals to 0. Otherwise it returns false. Also
     *         if any given arguments is null then it returns false.
     */
    protected boolean st(Q data0, Q data1) {
        return data0 != null && data1 != null && comparator.compare(data0, data1) < 0;
    }

    /**
     * @param data comparable object
     * @param src  source
     * @return true if comparison result equals to 0. Otherwise it returns false. Also
     *         if any given arguments is null then it returns false.
     */
    protected boolean gt(AVLNode<Q> src, AVLNode<Q> tar) {
        return src != null && tar != null && comparator.compare(src.getData(), tar.getData()) > 0;
    }

    /**
     * @param data comparable object
     * @param src  source
     * @return true if comparison result greater than 0. Otherwise it returns false. Also
     *         if any given arguments is null then it returns false.
     */
    protected boolean gt(Q data, AVLNode<Q> src) {
        return data != null && src != null && comparator.compare(data, src.getData()) > 0;
    }

    /**
     * @param data comparable object
     * @param src  source
     * @return true if comparison result equals to 0. Otherwise it returns false. Also
     *         if any given arguments is null then it returns false.
     */
    protected boolean gt(Q data0, Q data1) {
        return data0 != null && data1 != null && comparator.compare(data0, data1) > 0;
    }

    /**
     * @param src source
     * @return true if given nodes left child is empty. Otherwise it returns false. Also
     *         if given node is null then it returns false.
     */
    protected boolean le(AVLNode<Q> src) {
        return src != null && src.getLeft() == null;
    }

    /**
     * @param src source
     * @return true if given nodes right child is empty. Otherwise it returns false. Also
     *         if given node is null then it returns false.
     */
    protected boolean re(AVLNode<Q> src) {
        return src != null && src.getRight() == null;
    }

    /**
     * @param src source
     * @return true if given nodes left and right child is empty(leaf). Otherwise it returns false. Also
     *         if given node is null then it returns false.
     */
    protected boolean lf(AVLNode<Q> src) {
        return src != null && src.getLeft() == null && src.getRight() == null;
    }

    /**
     * @param node
     * @return
     */
    protected int height(AVLNode<Q> node) {
        if (node == null) {
            return -1;
        }

        return node.getHeight();
    }

    /**
     * @param node
     */
    protected void reheight(AVLNode<Q> node) {
        if (nonNull(node)) {

            int hg = 1 + Math.max(height(node.getLeft()), height(node.getRight()));
            node.setHeight(hg);

        }
    }

    /**
     * @return
     */
    public Comparator<Q> getComparator() {
        return comparator;
    }

    /**
     * @return
     */
    public AVLNode<Q> getRoot() {
        return root;
    }

    /**
     * @param root
     */
    public void setRoot(AVLNode<Q> root) {
        this.root = root;
    }
}
