/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-algorithm project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.algorithm;

import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.NoSuchElementException;
import terraml.algorithm.node.AStarNode;

/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
public class AStarGraph<Q> implements Iterable<Q> {

    /*
     * A map from the nodeId to outgoing edge.
     * An outgoing edge is represented as a tuple of NodeData and the edge length
     */
    private final Map<Q, Map<AStarNode<Q>, Double>> graph;
    /*
     * A map of heuristic from a node to each other node in the graph.
     */
    private final Map<Q, Map<Q, Double>> heuristicMap;
    /*
     * A map between nodeId and nodedata.
     */
    private final Map<Q, AStarNode<Q>> collector;

    /**
     * @param heuristicMap
     */
    public AStarGraph(Map<Q, Map<Q, Double>> heuristicMap) {
        if (heuristicMap == null) {
            throw new NullPointerException("The huerisic map should not be null");
        }

        graph = new HashMap<>();
        collector = new HashMap<>();
        this.heuristicMap = heuristicMap;
    }

    /**
     * Adds a new node to the graph.
     * Internally it creates the nodeData and populates the heuristic map concerning input node into node data.
     *
     * @param id the node to be added
     */
    public void addNode(Q id) {
        if (id == null) {
            throw new NullPointerException("The node cannot be null");
        }

        if (!heuristicMap.containsKey(id)) {
            throw new NoSuchElementException("This node is not a part of hueristic map");
        }

        graph.put(id, new HashMap<>());
        collector.put(id, new AStarNode<>(id, heuristicMap.get(id)));
    }

    /**
     * Adds an edge from source node to destination node.
     * There can only be a single edge from source to node.
     * Adding additional edge would overwrite the value
     *
     * @param nodeIdFirst  the first node to be in the edge
     * @param nodeIdSecond the second node to be second node in the edge
     * @param length       the length of the edge.
     */
    public void addEdge(Q nodeIdFirst, Q nodeIdSecond, double length) {
        if (nodeIdFirst == null || nodeIdSecond == null) {
            throw new NullPointerException("The first nor second node can be null.");
        }

        if (!heuristicMap.containsKey(nodeIdFirst) || !heuristicMap.containsKey(nodeIdSecond)) {
            throw new NoSuchElementException("Source and Destination both should be part of the part of hueristic map");
        }

        if (!graph.containsKey(nodeIdFirst) || !graph.containsKey(nodeIdSecond)) {
            throw new NoSuchElementException("Source and Destination both should be part of the part of graph");
        }

        graph.get(nodeIdFirst).put(collector.get(nodeIdSecond), length);
        graph.get(nodeIdSecond).put(collector.get(nodeIdFirst), length);
    }

    /**
     * Returns immutable view of the edges
     *
     * @param id the nodeId whose outgoing edge needs to be returned
     * @return An immutable view of edges leaving that node
     */
    public Map<AStarNode<Q>, Double> edgesFrom(Q id) {
        if (id == null) {
            throw new NullPointerException("The input node should not be null.");
        }

        if (!heuristicMap.containsKey(id)) {
            throw new NoSuchElementException("This node is not a part of hueristic map");
        }

        if (!graph.containsKey(id)) {
            throw new NoSuchElementException("The node should not be null.");
        }

        return Collections.unmodifiableMap(graph.get(id));
    }

    /**
     * The nodedata corresponding to the current nodeId.
     *
     * @param id the nodeId to be returned
     * @return the nodeData from the
     */
    public AStarNode<Q> getNodeData(Q id) {
        if (id == null) {
            throw new NullPointerException("The nodeid should not be empty");
        }

        if (!collector.containsKey(id)) {
            throw new NoSuchElementException("The nodeId does not exist");
        }

        return collector.get(id);
    }

    @Override
    public Iterator<Q> iterator() {
        return graph.keySet().iterator();
    }
}
