/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-algorithm project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.algorithm;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.PriorityQueue;
import java.util.Queue;
import java.util.Set;
import terraml.algorithm.node.AStarNode;
import terraml.commons.annotation.Reference;

/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 *
 */
public class AStar<Q> {

    private final AStarGraph<Q> graph;

    /**
     * @param graphAStar
     */
    public AStar(AStarGraph<Q> graphAStar) {
        this.graph = graphAStar;
    }

    public class NodeComparator implements Comparator<AStarNode<Q>> {

        /**
         * @param nodeFirst
         * @param nodeSecond
         * @return
         */
        public int compare(AStarNode<Q> nodeFirst, AStarNode<Q> nodeSecond) {
            if (nodeFirst.getCost() > nodeSecond.getCost()) {
                return 1;
            }

            if (nodeSecond.getCost() > nodeFirst.getCost()) {
                return -1;
            }

            return 0;
        }
    }

    /**
     * Implements the A-star algorithm and returns the path from source to destination
     *
     * @param source      the source nodeid
     * @param destination the destination nodeid
     * @return the path from source to destination
     */
    @Reference(
            implNote = "Why and how we used PriorityQueue",
            link = "http://stackoverflow.com/questions/20344041/why-does-priority-queue-has-default-initial-capacity-of-11"
    )
    public List<Q> astar(Q source, Q destination) {

        final Queue<AStarNode<Q>> openQueue = new PriorityQueue<>(11, new NodeComparator());

        AStarNode<Q> sourceNodeData = graph.getNodeData(source);
        sourceNodeData.setDistanceFromSource(0);
        sourceNodeData.calcF(destination);
        openQueue.add(sourceNodeData);

        final Map<Q, Q> path = new HashMap<>();
        final Set<AStarNode<Q>> closedList = new HashSet<>();

        while ( !openQueue.isEmpty() ) {

            final AStarNode<Q> nodeData = openQueue.poll();

            if (nodeData.getId().equals(destination)) {
                return path(path, destination);
            }

            closedList.add(nodeData);

            for ( Map.Entry<AStarNode<Q>, Double> neighborEntry : graph.edgesFrom(nodeData.getId()).entrySet() ) {

                AStarNode<Q> neighbor = neighborEntry.getKey();

                if (closedList.contains(neighbor)) {
                    continue;
                }

                double distanceBetweenTwoNodes = neighborEntry.getValue();
                double tentativeG = distanceBetweenTwoNodes + nodeData.getDistanceFromSource();

                if (tentativeG < neighbor.getDistanceFromSource()) {

                    neighbor.setDistanceFromSource(tentativeG);
                    neighbor.calcF(destination);

                    path.put(neighbor.getId(), nodeData.getId());

                    if (!openQueue.contains(neighbor)) {
                        openQueue.add(neighbor);
                    }

                }

            }

        }

        return null;
    }

    /**
     * @param path
     * @param destination
     * @return
     */
    private List<Q> path(Map<Q, Q> path, Q destination) {
        assert path != null;
        assert destination != null;

        final List<Q> pathList = new ArrayList<>();
        pathList.add(destination);

        while ( path.containsKey(destination) ) {
            destination = path.get(destination);
            pathList.add(destination);
        }

        Collections.reverse(pathList);
        return pathList;
    }
}
