package online.sanen.sendEmail;

import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.HashMap;
import java.util.Map;

import com.mhdt.analyse.Validate;
import com.mhdt.io.FileIO;
import com.mhdt.toolkit.Bean;

/**
 * 邮件实体Bean
 * <ul>
 * <li>{@link Mail#setTo(String)}收件人可以是多个,支持多个符号(;/；/，/,/空格) 隔开</li>
 * <li>{@link Mail#setNick(String)}个性名称设置nick值,这会让昵称显示在发件人上</li>
 * <li>{@link Mail#setPort(int)}默认端口号是25</li>
 * <li>{@link Mail#setAccessorypath(String)}支持附件路径</li>
 * <li>{@link Mail#setContent(String)} 设置文本时可以让字符串[file@文件路径]来链接文件</li>
 * <li>{@link Mail#setParam(String, String)} 如果想动态生成文本且在内容中使用<code>{\@+参数名}</code>格式预设参数,然后调用该方法给参数赋值</li>
 * </ul>
 * 
 * @author：懒得出风头 <br>
 * Date: 2016-05-13 <br>
 * Time: 上午10:14:05
 */

public class Mail {

	private String from;
	private String username;
	private String password;

	/** 昵称 */
	private String nick = "";

	/** Port */
	private int port = 25;

	/** SMTP Server */
	private String host;
	private String to;
	private String subject;
	private String content;

	/** 附件路径 */
	private String accessorypath;

	private Map<String, String> params = new HashMap<String, String>();

	public Mail() {

	}

	public Mail(String username, String password, String host) {
		this.username = username;
		this.from = username;
		this.password = password;
		this.host = host;

	}

	/**
	 * 加载配置文件可参考下列:
	 * <ul>
	 * <li>from = youmail@qq.com </li>
	 *<li> username = youmail@qq.com </li>
	 * <li>password = 123456 </li>
	 *<li> nick = Sanen Service </li>
	 * <li>host = smtp.qq.com</li>
	 * <li>port = 587 </li>
	 * <li>to =xxxx@qq.com,xxxx@qq.com,xxxx@qq.com </li>
	 * <li>subject = 测试主题 </li>
	 * </ul>
	 * @param config - 配置文件
	 * @throws IOException
	 */
	public Mail(File config) throws IOException {
		com.mhdt.parse.Properties pro = new com.mhdt.parse.Properties();
		pro.load(config);

		Mail email = Bean.parse(Mail.class, pro.map());
		this.username = email.username;
		this.password = email.password;
		this.from = email.from;
		this.nick = Validate.isNullOrEmpty(email.nick) ? this.nick : email.nick;
		this.port = Validate.isNullOrEmpty(email.port) ? this.port : email.port;
		this.host = email.host;
		this.to = email.to;
		this.subject = email.subject;
		this.content = email.content;

	}

	public String getPassword() {
		return password;
	}

	public void setPassword(String password) {
		this.password = password;
	}

	public String getTo() {
		return to;
	}

	public void setTo(String to) {
		this.to = to;
	}

	public String getSubject() {
		return subject;
	}

	public void setSubject(String subject) {
		this.subject = subject;
	}

	public String getContent() {
		String temp = new String(this.content);

		for (Map.Entry<String, String> entry : params.entrySet()) {
			temp = temp.replace("${" + entry.getKey() + "}", entry.getValue());
		}

		return temp;
	}

	/**
	 * <pre>
	 * 内容可以使用<code>{\@参数名}</code>标签
	 * 1.字符串
	 * 2.file@绑定文件文本内容
	 * &#64;param content
	 * </pre>
	 */
	public void setContent(String content) {
		this.content = content;

		if (!Validate.isNullOrEmpty(this.content) && this.content.startsWith("file@")) {
			File file = new File(this.content.substring(content.indexOf("@") + 1));
			this.content = (FileIO.getContent(file));
		}
	}

	/** 获取附件路径 */
	public String getAccessorypath() {
		return accessorypath;
	}

	/** 设置附件路径 */
	public void setAccessorypath(String accessorypath) {
		this.accessorypath = accessorypath;
	}

	public String getHost() {
		return host;
	}

	public void setHost(String host) {
		this.host = host;
	}


	public int getPort() {
		return port;
	}

	public void setPort(int port) {
		this.port = port;
	}

	public String getUsername() {
		return username;
	}

	public void setUsername(String username) {
		this.username = username;
	}

	public String getNick() {
		return nick;
	}

	public void setNick(String nick) {
		try {
			nick = javax.mail.internet.MimeUtility.encodeText("sanen Service");
		} catch (UnsupportedEncodingException e) {
			e.printStackTrace();
			nick = "";
		}

		this.nick = nick;
	}

	public String getFrom() {
		return from;
	}

	public void setFrom(String from) {
		this.from = from;
	}
	
	
	public void setParam(String param, String value) {
		params.put(param, value);
	}

	public void clearParams() {
		params.clear();
	}

	@Override
	public String toString() {
		return "Mail [from=" + from + "\n, " + "username=" + username + "\n, " + "password=" + password + "\n, "
				+ "port=" + port + "\n," + " host=" + host + "\n," + " to=" + to + "\n, " + "subject=" + subject
				+ "\n, " + "content=" + content + "\n, " + "accessorypath=" + accessorypath + "]";
	}

}
