package online.sanen.cdm.factory;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Consumer;

import com.mhdt.analyse.Validate;
import com.mhdt.toolkit.Assert;

import online.sanen.cdm.BootstrapDevice;
import online.sanen.cdm.api.Bootstrap;
import online.sanen.cdm.api.basic.SimpleConfiguration;
import online.sanen.cdm.api.basic.Configuration.DataSouseType;
import online.sanen.cdm.api.basic.CdmStructuralException;
import online.sanen.cdm.api.basic.DriverOption;
import online.sanen.cdm.api.basic.FileConfiguration;
import online.sanen.cdm.template.transaction.Transaction;

/**
 * A {@link Bootstrap} instance is quickly created to replace the <b>new</b>
 * method, and it also has certain container functions
 * 
 * @author LazyToShow <br>
 *         Date: 2018/06/12 <br>
 *         Time: 09:17
 */
public class Bootstraps {

	static Map<Object, Bootstrap> exists = new HashMap<>();

	public static boolean contains(Object key) {
		return exists.containsKey(key);
	}

	public static Bootstrap get(Object key) {

		return exists.get(key);
	}

	/**
	 * 
	 * @param bootStrapId
	 * @param bootstrap
	 * @return
	 */
	public static Bootstrap put(Object bootStrapId, Bootstrap bootstrap) {

		exists.put(bootStrapId, bootstrap);
		return bootstrap;

	}

	/**
	 * 
	 * @param bootstrapId
	 * @return
	 */
	public static Bootstrap remove(String bootstrapId) {
		return exists.remove(bootstrapId);
	}

	public static void removeByPrefix(String prefix) {

		List<String> array = new ArrayList<>();

		exists.keySet().forEach(item -> {
			if (item.toString().startsWith(prefix))
				array.add(item.toString());
		});

		array.forEach(exists::remove);
	}

	public static Bootstrap load(Consumer<SimpleConfiguration> consumer) {
		return load(null, consumer);
	}

	public static Bootstrap loadByFileConfiguration(Consumer<FileConfiguration> consumer) {
		return loadByFileConfiguration(null, consumer);
	}

	public static Bootstrap loadByFileConfiguration(Object bootStrapId, Consumer<FileConfiguration> consumer) {

		try {

			if (bootStrapId != null && exists.containsKey(bootStrapId))
				return exists.get(bootStrapId);

			FileConfiguration configuration = new FileConfiguration();

			consumer.accept(configuration);

			Bootstrap bootstrap = new BootstrapDevice(configuration, bootStrapId);

			if (bootstrap != null)
				exists.put(bootStrapId, bootstrap);

			return bootstrap;

		} catch (Exception e) {
			throw new CdmStructuralException(e);
		}

	}

	public static Bootstrap load(Object bootStrapId, Consumer<SimpleConfiguration> configuration) {

		try {

			if (bootStrapId != null && exists.containsKey(bootStrapId))
				return exists.get(bootStrapId);

			SimpleConfiguration conf = new SimpleConfiguration();

			configuration.accept(conf);

			Bootstrap bootstrap = new BootstrapDevice(conf, bootStrapId);

			if (bootstrap != null)
				exists.put(bootStrapId, bootstrap);

			return bootstrap;

		} catch (Exception e) {
			throw new CdmStructuralException(e);
		}

	}

	/**
	 * Whether the BootStrap instance is unique
	 * 
	 * @return boolean
	 */
	public static boolean isUniqueness() {
		return exists.size() == 1;
	}

	/**
	 * Returns the first {@link Bootstrap} instance, or null if the cache is empty.
	 * 
	 * @return {@link Bootstrap}
	 */
	public static Bootstrap getFirst() {
		Optional<Object> findFirst = exists.keySet().stream().findFirst();
		return findFirst.isPresent() ? exists.get(findFirst.get()) : null;
	}

	static Map<Bootstrap, Transaction> transactions = new HashMap<>();

	/**
	 * 
	 * @param bootstrap
	 * @param transaction
	 */
	public static void registedTransaction(Bootstrap bootstrap, Transaction transaction) {

		transactions.put(bootstrap, transaction);
	}

	public static BootstrapBuilder newBuilder(String bootstrapId) {
		return new BootstrapBuilder(bootstrapId);
	}

	public static class BootstrapBuilder {

		private Object bootStrapId;
		private String driver;
		private String username;
		private String password;
		private Boolean isShowSql;
		private String url;
		private String validationQuery;
		private DataSouseType dataSouseType;
		private Boolean isFormat;
		private Boolean isCache;

		private BootstrapBuilder(String bootstrapId2) {
			
			Assert.notNull(bootstrapId2, "bootstrap cannot be null");
			this.bootStrapId = bootstrapId2;
		}

		public BootstrapBuilder setBootstrapId(Object bootStrapId) {
			this.bootStrapId = bootStrapId;
			return this;
		}

		public BootstrapBuilder setDriver(DriverOption driverOption) {
			this.driver = driverOption.getValue();
			return this;
		}

		public BootstrapBuilder setUsername(String username) {
			this.username = username;
			return this;
		}

		public BootstrapBuilder setPassword(String password) {
			this.password = password;
			return this;
		}

		public BootstrapBuilder setShowSql(boolean isShowSql) {
			this.isShowSql = isShowSql;
			return this;
		}

		public BootstrapBuilder setIsCache(boolean isCache) {
			this.isCache = isCache;
			return this;
		}

		public BootstrapBuilder setIsFormat(boolean isFormat) {
			this.isFormat = isFormat;
			return this;
		}

		/**
		 * Set the JDBC connection url
		 * 
		 * - Enumerate the different database connections:
		 * 
		 *            <pre>
		 * 				<code>
		 *            jdbc:oracle:thin:@//127.0.0.1:1521/orcl
		 *            jdbc:mysql://127.0.0.1:3306/test?useSSL=false
		 *            jdbc:sqlite:tmp/data/test.sqlite
		 *            jdbc:sqlserver://127.0.0.1:1433;DatabaseName=test
		 *        		 </code>
		 *            </pre>
		 */
		public BootstrapBuilder setUrl(String url) {
			this.url = url;
			return this;
		}

		public BootstrapBuilder setValidationQuery(String validationQuery) {
			this.validationQuery = validationQuery;
			return this;
		}

		public BootstrapBuilder setDataSouseType(DataSouseType dataSouseType) {
			this.dataSouseType = dataSouseType;
			return this;
		}

		public Bootstrap build() {

			return Bootstraps.load(bootStrapId, configuration -> {

				if (!Validate.isNullOrEmpty(driver))
					configuration.setDriverOption(driver);

				if (!Validate.isNullOrEmpty(username))
					configuration.setUsername(username);

				if (!Validate.isNullOrEmpty(password))
					configuration.setPassword(password);

				if (!Validate.isNullOrEmpty(isShowSql))
					configuration.setShowSql(isShowSql);

				if (!Validate.isNullOrEmpty(isCache))
					configuration.setShowSql(isCache);

				if (!Validate.isNullOrEmpty(isFormat))
					configuration.setShowSql(isFormat);

				if (!Validate.isNullOrEmpty(url))
					configuration.setUrl(url);

				if (!Validate.isNullOrEmpty(validationQuery))
					configuration.setValidationQuery(validationQuery);

				if (!Validate.isNullOrEmpty(dataSouseType))
					configuration.setDataSouseType(dataSouseType);

			});

		}

	}

}
