package online.sanen.cdm;

import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import java.util.function.Function;

import online.sanen.cdm.basic.BasicBean;
import online.sanen.cdm.basic.CdmConditionException;
import online.sanen.cdm.basic.DataField;
import online.sanen.cdm.basic.ProductType;
import online.sanen.cdm.basic.QueryType;
import online.sanen.cdm.basic.ResultType;
import online.sanen.cdm.basic.Sorts;
import online.sanen.cdm.basic.StreamConsumer;
import online.sanen.cdm.basic.Structure;
import online.sanen.cdm.basic.Structure.SortSupport;
import online.sanen.cdm.component.Manager;
import online.sanen.cdm.component.Pipeline;
import online.sanen.cdm.component.PipelineDivice;
import online.sanen.cdm.condition.Condition;
import online.sanen.cdm.condition.Condition.Conditions;
import online.sanen.cdm.factory.HandelFactory;
import online.sanen.cdm.factory.PipelineFactory;
import online.sanen.cdm.template.JdbcTemplate;

/**
 * 
 * @author LazyToShow <br>
 *         Date: 2017/11/23 <br>
 *         Time: 22:22
 */
public class QueryTBDevice implements QueryTB {

	Structure structure;

	public QueryTBDevice(Manager manager, String tableName) {
		structure = new Structure(manager);
		structure.setTableName(tableName);
	}

	@Override
	public QueryTB addEntry(Class<? extends BasicBean> entry) {
		structure.setEntry_class(entry);
		return this;
	}

	@Override
	public QueryTB addCondition(String fieldName, Conditions conditions) {
		try {
			structure.addCondition(Condition.builder(fieldName, conditions));
		} catch (CdmConditionException e) {
			e.printStackTrace();
		}
		return this;
	}

	@Override
	public QueryTB addCondition(String fieldName, Conditions conditions, Object value) {
		structure.addCondition(Condition.builder(fieldName, conditions, value));
		return this;
	}

	@Override
	public QueryTB addCondition(Consumer<List<Condition>> consumer) {
		consumer.accept(structure.getConditions());
		return this;
	}

	@Override
	public QueryTB sort(final Sorts sorts, final String... fields) {

		structure.setSortSupport(new SortSupport() {

			@Override
			public String toString() {

				StringBuilder sb = new StringBuilder(" order by ");

				for (String field : fields) {
					sb.append(field + ",");
				}

				sb.setLength(sb.length() - 1);
				sb.append(" " + sorts.getValue());

				return sb.toString();
			}
		});

		return this;
	}

	@Override
	public QueryTB setFields(String... fields) {
		structure.setFields(new HashSet<>(Arrays.asList(fields)));
		return this;
	}

	@Override
	public QueryTB setExceptFields(String... fields) {
		structure.setExceptes(new HashSet<>(Arrays.asList(fields)));
		return this;
	}

	@SuppressWarnings("unchecked")
	@Override
	public <T> T unique() {
		return (T) Assembler.create(QueryType.select, ResultType.Object, structure, new PipelineFactory() {
			@Override
			public Pipeline getPipeline() {
				Pipeline pipeline = new PipelineDivice();
				pipeline.addLast(HandelFactory.commonFieldHandel());
				pipeline.addLast(HandelFactory.sqlHandel());
				pipeline.addLast(HandelFactory.conditionHandel());
				pipeline.addLast(HandelFactory.paramerHandel());
				pipeline.addLast(HandelFactory.resultHandel());
				pipeline.addLast(HandelFactory.debugHandel());
				return pipeline;
			}
		});
	}

	@SuppressWarnings("unchecked")
	@Override
	public <T> List<T> list() {
		return (List<T>) Assembler.create(QueryType.select, ResultType.List, structure, new PipelineFactory() {
			@Override
			public Pipeline getPipeline() {
				Pipeline pipeline = new PipelineDivice();
				pipeline.addLast(HandelFactory.commonFieldHandel());
				pipeline.addLast(HandelFactory.sqlHandel());
				pipeline.addLast(HandelFactory.conditionHandel());
				pipeline.addLast(HandelFactory.paramerHandel());
				pipeline.addLast(HandelFactory.limitHandel());
				pipeline.addLast(HandelFactory.resultHandel());
				pipeline.addLast(HandelFactory.debugHandel());
				return pipeline;
			}
		});
	}
	
	

	@Override
	public void stream(int bufferSize,Consumer<List<Map<String,Object>>> consumer) {

		Assembler.create(QueryType.select, ResultType.Maps, structure, new PipelineFactory() {

			@Override
			public Pipeline getPipeline() {
				Pipeline pipeline = new PipelineDivice();
				pipeline.addLast(HandelFactory.commonFieldHandel());
				pipeline.addLast(HandelFactory.sqlHandel());
				pipeline.addLast(HandelFactory.conditionHandel());
				pipeline.addLast(HandelFactory.paramerHandel());
				pipeline.addLast(HandelFactory.streamHandel(bufferSize,consumer,null));
				return pipeline;
			}
		});

	}
	
	@Override
	public void stream(int bufferSize, Consumer<List<Map<String, Object>>> consumer,Map<String,String> aliases) {

		Assembler.create(QueryType.select, ResultType.Maps, structure, new PipelineFactory() {

			@Override
			public Pipeline getPipeline() {
				Pipeline pipeline = new PipelineDivice();
				pipeline.addLast(HandelFactory.commonFieldHandel());
				pipeline.addLast(HandelFactory.sqlHandel());
				pipeline.addLast(HandelFactory.conditionHandel());
				pipeline.addLast(HandelFactory.paramerHandel());
				pipeline.addLast(HandelFactory.streamHandel(bufferSize, consumer, aliases));
				return pipeline;
			}
		});

	}
	
	@Override
	public void stream(int bufferSize, Function<List<DataField>, Object> consumer, StreamConsumer datas,
			Map<String, String> aliases) {
		
		Assembler.create(QueryType.select, ResultType.Maps, structure, new PipelineFactory() {

			@Override
			public Pipeline getPipeline() {
				Pipeline pipeline = new PipelineDivice();
				pipeline.addLast(HandelFactory.commonFieldHandel());
				pipeline.addLast(HandelFactory.sqlHandel());
				pipeline.addLast(HandelFactory.conditionHandel());
				pipeline.addLast(HandelFactory.paramerHandel());
				pipeline.addLast(HandelFactory.streamHandel(bufferSize, consumer,datas, aliases));
				return pipeline;
			}
		});
		
	}


	@SuppressWarnings("unchecked")
	@Override
	public List<Map<String, Object>> maps() {

		return (List<Map<String, Object>>) Assembler.create(QueryType.select, ResultType.Maps, structure,
				new PipelineFactory() {
					@Override
					public Pipeline getPipeline() {
						Pipeline pipeline = new PipelineDivice();
						pipeline.addLast(HandelFactory.commonFieldHandel());
						pipeline.addLast(HandelFactory.sqlHandel());
						pipeline.addLast(HandelFactory.conditionHandel());
						pipeline.addLast(HandelFactory.paramerHandel());
						pipeline.addLast(HandelFactory.limitHandel());
						pipeline.addLast(HandelFactory.resultHandel());
						pipeline.addLast(HandelFactory.debugHandel());
						return pipeline;
					}
				});
	}

	@Override
	public int delete() {
		return (int) Assembler.create(QueryType.delete, ResultType.Int, structure, new PipelineFactory() {
			@Override
			public Pipeline getPipeline() {
				Pipeline pipeline = new PipelineDivice();
				pipeline.addLast(HandelFactory.sqlHandel());
				pipeline.addLast(HandelFactory.conditionHandel());
				pipeline.addLast(HandelFactory.paramerHandel());
				pipeline.addLast(HandelFactory.resultHandel());
				pipeline.addLast(HandelFactory.debugHandel());
				return pipeline;
			}
		});
	}

	@Override
	public QueryTB limit(Integer... args) {
		structure.setLimit(args);
		structure.setHasLimitAble(true);
		return this;
	}

	@Override
	public boolean isExsites() {
		try {
			JdbcTemplate template = (JdbcTemplate) structure.getTemplate();
			template.queryForRowSet("SELECT 1 FROM " + structure.getTableName());
			return true;
		} catch (Exception e) {
			return false;
		}

	}

	@Override
	public int updateName(String newName) {

		structure.setSql(ProductType.updateTableNameSQL(structure.productType(), structure.getTableName(), newName));

		return (int) Assembler.create(QueryType.update, ResultType.Int, structure, new PipelineFactory() {
			@Override
			public Pipeline getPipeline() {
				Pipeline pipeline = new PipelineDivice();
				pipeline.addLast(HandelFactory.resultHandel());
				pipeline.addLast(HandelFactory.debugHandel());
				return pipeline;
			}
		});

	}

	@Override
	public int clear() {
		return delete();
	}

	@Override
	public int drop() {
		

		return (int) Assembler.create(QueryType.drop, ResultType.Int, structure, new PipelineFactory() {
			@Override
			public Pipeline getPipeline() {
				Pipeline pipeline = new PipelineDivice();
				pipeline.addLast(HandelFactory.sqlHandel());
				pipeline.addLast(HandelFactory.resultHandel());
				pipeline.addLast(HandelFactory.debugHandel());
				return pipeline;
			}
		});
	}

	@Override
	public int addColumn(String columnName, String type) {

		structure.setSql("ALTER TABLE " + structure.getTableName() + " ADD COLUMN  " + columnName + " " + type);

		return (int) Assembler.create(QueryType.update, ResultType.Int, structure, new PipelineFactory() {
			@Override
			public Pipeline getPipeline() {
				Pipeline pipeline = new PipelineDivice();
				pipeline.addLast(HandelFactory.resultHandel());
				pipeline.addLast(HandelFactory.debugHandel());
				return pipeline;
			}
		});

	}

	@Override
	public String createAndInsert(String newTableName) {
		return  (String) Assembler.create(QueryType.select, ResultType.String, structure, new PipelineFactory() {
			@Override
			public Pipeline getPipeline() {
				Pipeline pipeline = new PipelineDivice();
				pipeline.addLast(HandelFactory.commonFieldHandel());
				pipeline.addLast(HandelFactory.sqlHandel());
				pipeline.addLast(HandelFactory.conditionHandel());
				pipeline.addLast(HandelFactory.paramerHandel());
				pipeline.addLast(HandelFactory.limitHandel());
				pipeline.addLast(HandelFactory.createAndInsert(newTableName));
				pipeline.addLast(HandelFactory.debugHandel());
				return pipeline;
			}
		});
	}
	

}
