package online.sanen.cdm;

import java.util.List;
import java.util.function.Consumer;

import com.mhdt.analyse.Validate;
import com.mhdt.toolkit.Assert;
import com.mhdt.toolkit.Reflect;

import online.sanen.cdm.basic.BasicBean;
import online.sanen.cdm.basic.Cdm;
import online.sanen.cdm.basic.CdmQueryException;
import online.sanen.cdm.condition.Condition;
import online.sanen.cdm.condition.Condition.Conditions;
import online.sanen.cdm.factory.BootStrapFactoty;

/**
 * As the name implies, this interface is used to add some behavior to an
 * object. You can think about this question: is an object without behavior a
 * real object?<br>
 * <br>
 * 
 * Injecting localization into the Model itself requires only that the Model
 * implements the interface.<br>
 * <br>
 * 
 * This interface is not a complete implementation, but is built for business
 * independence, in such a way that you can run and debug your project without
 * having to rely on the container.
 * 
 * @author LazyToShow <br>
 *         Date： 2018年7月20日 <br>
 *         Time: 下午4:27:45
 */
public interface Behavior extends BasicBean {

	/**
	 * Get the {@link Bootstrap} instance corresponding to the current model. Note
	 * that new {@link Bootstrap} should not be constructed here, but should be
	 * obtained from the created object pool or cache container.
	 * 
	 * @return
	 */
	default Bootstrap bootstrap() {

		if (BootStrapFactoty.isUniqueness())
			return BootStrapFactoty.getFirst();

		String bootstrapId = Reflect.getBootStrapId(this.getClass());

		Assert.notNull(bootstrapId, "class:" + this.getClass().getName()
				+ " There is no injected bootstrap instance, please use the bootstrap annotation or rewrite the getBootstrap method.");

		if (BootStrapFactoty.contains(bootstrapId))
			return BootStrapFactoty.get(bootstrapId);
		else
			throw new NullPointerException("BootstrapId: '" + bootstrapId
					+ "' index instance cannot be passed by class:" + this.getClass().getName());

	}

	/**
	 * Inserts the current instance to the database
	 * 
	 * @return
	 */
	default int insert() {
		return bootstrap().query(this).insert();
	}

	/**
	 * Delete in the database according to the {@link #primarykey()}
	 * 
	 * @return
	 */
	default int delete() {
		return bootstrap().query(this).delete();
	}

	/**
	 * Update in the database according to the {@link #primarykey()}
	 * 
	 * @return
	 */
	default int update() {
		return bootstrap().query(this).update();
	}

	/**
	 * Update in the database according to the {@link #primarykey()}
	 * 
	 * @param fields - Specify which fields need to be modified
	 * @return
	 */
	default int updateByFields(String... fields) {
		return bootstrap().query(this).setFields(fields).update();
	}

	/**
	 * Update in the database according to the {@link #primarykey()}
	 * 
	 * @param fields - Specify fields that do not need to be modified
	 * @return
	 */
	default int updateByExceptFields(String... fields) {
		return bootstrap().query(this).setExceptFields(fields).update();
	}

	/**
	 * Gets an instance from the database through the primary key
	 * 
	 * @return
	 */
	@SuppressWarnings("unchecked")
	default <T extends Behavior> T findByPk() {
		try {

			Object primary = Reflect.getValue(this, primarykey() == null ? "id" : primarykey());
			return (T) bootstrap().query(this.getClass(), primary).find();

		} catch (IllegalArgumentException | IllegalAccessException | NoSuchFieldException | SecurityException e) {
			e.printStackTrace();
		}

		return null;
	}

	/**
	 * 
	 * @param conditions
	 * @return
	 */
	default <T extends Behavior> T findByConditions(Consumer<List<Condition>> conditions) {

		return bootstrap().query(this.getClass()).addCondition(conditions).addEntry(this.getClass()).unique();

	}

	/**
	 * 
	 * @param fields
	 * @return
	 */
	default <T extends Behavior> T findByField(String... fields) {

		return bootstrap().query(this.getClass()).addCondition(conditions -> {

			for (String fieldName : fields) {

				if (Validate.hasField(this.getClass(), fieldName))
					try {
						conditions.add(
								Condition.builder(fieldName, Conditions.EQUALS, Reflect.getValue(this, fieldName)));
					} catch (IllegalArgumentException | IllegalAccessException | NoSuchFieldException
							| SecurityException e) {
						e.printStackTrace();
					}
			}

		}).addEntry(this.getClass()).unique();

	}

	/**
	 * Create tables based on data structures
	 * 
	 * @return
	 */
	default int createTable() {

		if (!bootstrap().query(Cdm.tableName(this.getClass())).isExsites())
			return bootstrap().query(this).create();

		return -1;

	}

	/**
	 * Delete table
	 * 
	 * @return
	 */
	default int dropTable() {

		String tableName = Cdm.tableName(this.getClass());

		if (bootstrap().query(tableName).isExsites())
			return bootstrap().query(tableName).drop();

		return -1;
	}

	/**
	 * 
	 * @param cls
	 * @return
	 */
	public static <T extends Behavior> List<T> list(Class<T> cls) {

		Bootstrap bootstrap = BootStrapFactoty.isUniqueness() ? BootStrapFactoty.getFirst()
				: BootStrapFactoty.get(Reflect.getBootStrapId(cls));

		return bootstrap.query(cls).addEntry(cls).list();
	}

	/**
	 * 
	 * @param cls
	 * @param consumer
	 * @return
	 */
	public static <T extends Behavior> List<T> listByConditions(Class<T> cls, Consumer<List<Condition>> consumer) {

		Bootstrap bootstrap = BootStrapFactoty.isUniqueness() ? BootStrapFactoty.getFirst()
				: BootStrapFactoty.get(Reflect.getBootStrapId(cls));

		return bootstrap.query(cls).addEntry(cls).addCondition(consumer).list();
	}

	/**
	 * 
	 * @param t
	 * @param fields
	 * @return
	 */
	public static <T extends Behavior> List<T> listByFields(T t, String... fields) {

		Bootstrap bootstrap = BootStrapFactoty.isUniqueness() ? BootStrapFactoty.getFirst()
				: BootStrapFactoty.get(Reflect.getBootStrapId(t.getClass()));

		return bootstrap.query(t.getClass()).addEntry(t.getClass()).addCondition(conditions -> {

			for (String fieldName : fields) {

				if (Validate.hasField(t.getClass(), fieldName))
					try {
						conditions.add(Condition.builder(fieldName, Conditions.EQUALS, Reflect.getValue(t, fieldName)));
					} catch (IllegalArgumentException | IllegalAccessException | NoSuchFieldException
							| SecurityException e) {
						e.printStackTrace();
					}
			}

		}).list();
	}

	/**
	 * 
	 * @param list
	 * @return
	 * @throws CdmQueryException
	 */
	public static <T extends Behavior> int batchInsert(List<T> list) throws CdmQueryException {

		Bootstrap bootstrap = BootStrapFactoty.isUniqueness() ? BootStrapFactoty.getFirst()
				: BootStrapFactoty.get(Reflect.getBootStrapId(list.get(0).getClass()));

		if (list != null && !list.isEmpty())
			return bootstrap.query(list).insert();
		else
			throw new CdmQueryException("list is null or is Empty");
	}

	/**
	 * 
	 * @param cls
	 * @param conditions
	 * @return
	 * @throws CdmQueryException
	 */
	public static <T extends Behavior> int delete(Class<T> cls, Consumer<List<Condition>> conditions)
			throws CdmQueryException {

		Bootstrap bootstrap = BootStrapFactoty.isUniqueness() ? BootStrapFactoty.getFirst()
				: BootStrapFactoty.get(Reflect.getBootStrapId(cls));

		return bootstrap.query(cls).addCondition(conditions).delete();
	}

}
