package online.sanen.cdm.handel;

import java.lang.reflect.Field;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.support.rowset.SqlRowSet;

import com.mhdt.annotation.Column;
import com.mhdt.annotation.NoDB;
import com.mhdt.annotation.NoInsert;
import com.mhdt.annotation.NoUpdate;
import com.mhdt.toolkit.Reflect;

import online.sanen.cdm.Constant;
import online.sanen.cdm.Handel;
import online.sanen.cdm.basic.QueryType;
import online.sanen.cdm.basic.Structure;

/**
 * <pre>
 * Field processing, which takes the intersection of entry and database tables,
 * the type of which is the entry
 * 
 * &#64;author LazyToShow 
 * Date: 2017/10/21 
 * Time: 23:19
 * </pre>
 */
public class CommonFieldHandel implements Handel {

	static Map<String, HashSet<String>> map = new HashMap<String, HashSet<String>>();

	@Override
	public Object handel(Structure structure, Object product) throws SQLException {

		// If the fields are specified
		if (structure.getFields() != null && structure.getFields().size() > 0) {
			structure.setCommonFields(structure.getFields());
			return null;

		}

		// Get the table field
		String tableName = structure.getTableName().toUpperCase();

		// The query table has all the fields
		Set<String> tableFields = getTableFields(tableName, structure);
		Set<String> entryFields = getEntryField(structure);
		

		// Table fields and class fields are collected and set
		if (entryFields == null) {
			entryFields = tableFields;
		} else {
			entryFields.retainAll(tableFields);
		}

		structure.setCommonFields(entryFields);

		return null;
	}

	private Set<String> getEntryField(Structure structure) {
		Class<?> cls = structure.getEntry_class();
		if (cls == null)
			return null;

		QueryType queryType = structure.getQueryType();
		Set<String> columns = new HashSet<>();

		for (Field field : Reflect.getFields(cls)) {

			// Annotations to skip
			if (Reflect.hasAnnotation(field, NoDB.class))
				continue;

			// Annotations to skip
			if (queryType.equals(QueryType.update) && Reflect.hasAnnotation(field, NoUpdate.class))
				continue;

			// Annotations to skip
			if (queryType.equals(QueryType.insert) && Reflect.hasAnnotation(field, NoInsert.class))
				continue;

			// Annotations to skip
			if (structure.getExceptes() != null && structure.getExceptes().contains(field.getName()))
				continue;

			// The alias is preferred
			if (Reflect.hasAnnotation(field, Column.class)) {
				columns.add(Reflect.getColumnValue(field).toLowerCase());
				continue;
			}

			// Add field name
			columns.add(field.getName().toLowerCase());
		}

		return columns;

	}

	/**
	 * 
	 * @param tableName
	 * @param structure
	 * @return
	 * @throws SQLException 
	 */
	private HashSet<String> getTableFields(String tableName, Structure structure) throws SQLException {
		// If there is a cache, return directly
		if (map.containsKey(tableName)) {
			return map.get(tableName);
		}

		// Find all the fields of the table through Sql
		JdbcTemplate template = structure.getJdbcTemplate();
		String databaseName = structure.getDataBaseName();
		SqlRowSet rs = null;
		switch (databaseName) {

		case Constant.MYSQL:
			rs = getSqlColumns(template, tableName);
			break;
			
		case Constant.SQLITE:
			rs = getSqliteColumns(template, tableName);
			break;

		case Constant.MICROSOFT_SQL_SERVER:
			rs = getSqlServerColumns(template, tableName);
			break;

		case Constant.ORACLE:
			rs = getOracleColumns(template, tableName);
			break;

		default:
			throw new RuntimeException("You may not support the current connection database:" + databaseName
					+ " type, but you can use the createSQL interface to continue the operation.");
		}

		HashSet<String> result = new HashSet<>();
		while (rs.next()) {
			if(databaseName.equals(Constant.SQLITE)) {
				result.add(rs.getString(2).toLowerCase());
			}else {
				result.add(rs.getString(1).toLowerCase());
			}
				
		}

		// Unable to obtain the type database field to throw an exception
		if (result.isEmpty())
			throw new RuntimeException("Cannot read tables from  database.The reason might be table name is not exist: [" + tableName + "] Or no database connection. ");

		// Join the cache
		map.put(tableName, result);
		return result;
	}

	private SqlRowSet getSqliteColumns(JdbcTemplate template, String tableName) {
		try {
			String sql = "PRAGMA table_info('"+tableName+"')";
			return template.queryForRowSet(sql);
		}catch (Exception e) {
			e.printStackTrace();
		}
		return null;
	}

	private SqlRowSet getOracleColumns(JdbcTemplate template, String tableName) {
		try{
			String sql = "select COLUMN_NAME from dba_tab_columns where table_name =upper('"+tableName+"')";
			return template.queryForRowSet(sql);
		}catch (Exception e) {
			e.printStackTrace();
		}
		return null;
	}

	/**
	 * 
	 * @param template
	 * @param tableName
	 * @return
	 */
	private SqlRowSet getSqlServerColumns(JdbcTemplate template, String tableName) {
		try {
			String sql = "SELECT column_name FROM INFORMATION_SCHEMA.columns WHERE TABLE_NAME=? ";
			return template.queryForRowSet(sql, tableName);
		} catch (DataAccessException e) {
			e.printStackTrace();
		}
		return null;
	}

	/**
	 * 
	 * @param template
	 * @param tableName
	 * @return
	 */
	private SqlRowSet getSqlColumns(JdbcTemplate template, String tableName) {
		try {
			String sql = "SELECT COLUMN_NAME FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_NAME = ? AND TABLE_SCHEMA = ?;";
			return template.queryForRowSet(sql, tableName, template.getDataSource().getConnection().getCatalog());
		} catch (DataAccessException | SQLException e) {
			e.printStackTrace();
		}
		return null;
	}

}
