package online.sanen.cdm.api.basic;

import java.sql.Connection;
import java.util.HashSet;

import online.sanen.cdm.template.SqlTemplate;
import online.sanen.cdm.template.SqlRowSet;

/**
 * Database of different manufacturers enumeration, at the same time because of
 * different manufacturers caused differences, also try to resolve in this type
 * 
 * @author LazyToShow <br>
 *         Date： 2018年8月20日 <br>
 *         Time: 上午11:09:31
 */
public enum ProductType {

	MYSQL, SQLITE, MICROSOFT_SQL_SERVER, ORACLE;

	/**
	 * In SQL queries, if the table name contains a special symbol that needs to be
	 * wrapped in a layer of modifiers, different database vendor modifiers will be
	 * different, and this method is designed to provide uniformly, and by default,
	 * return an empty string
	 * 
	 * @param productType
	 * @return
	 */
	public static String applyTableModifier(ProductType productType) {

		if (productType == ProductType.MYSQL) {
			return "`";
		} else if (productType == ProductType.SQLITE || productType == ProductType.ORACLE) {
			return "\"";
		} else {
			return "";
		}

	}

	/**
	 * 
	 * @param productType
	 * @param tableName
	 * @param newName
	 * @return
	 */
	public static String updateTableNameSQL(ProductType productType, String tableName, String schema, String newName) {

		String modifier = ProductType.applyTableModifier(productType);

		if (productType == ProductType.MICROSOFT_SQL_SERVER)
			return String.format("EXEC sp_rename %s%s%s,'%s'", modifier, tableName, modifier, newName);
		else if(schema!=null){
			return String.format("ALERT TABLE %s%s%s.%s%s%s RENAME TO %s%s%s", 
					modifier, schema, modifier, 
					modifier, tableName, modifier, 
					modifier, newName,modifier);
		}else {
			return String.format("ALERT TABLE %s%s%s RENAME TO %s%s%s", 
					modifier, tableName, modifier, 
					modifier, newName,modifier);
		}
			
	}

	/**
	 * 
	 * @param productType
	 * @param template
	 * @param tableName
	 * @return
	 */
	public static HashSet<String> getColumnsFromTableName(ProductType productType, SqlTemplate template,
			String tableName, String schema) {

		SqlRowSet rs = null;

		switch (productType) {

		case MYSQL:

			try {

				if (schema == null) {
					try (Connection conn = template.getDataSource().getConnection()) {
						schema = conn.getCatalog();
					}
				}

				rs = template.queryForRowSet(
						"SELECT COLUMN_NAME FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_NAME = ? AND TABLE_SCHEMA = ?;",
						tableName, schema);

			} catch (Exception e) {
				throw new CdmQueryException(e);
			}

			break;

		case SQLITE:
			rs = template.queryForRowSet("PRAGMA table_info('" + tableName + "')");
			break;

		case MICROSOFT_SQL_SERVER:
			rs = template.queryForRowSet("SELECT column_name FROM INFORMATION_SCHEMA.columns WHERE TABLE_NAME=? ",
					tableName);
			break;

		case ORACLE:
			if (tableName.contains("."))
				tableName = tableName.split("\\.")[1];
			rs = template.queryForRowSet("SELECT t.COLUMN_NAME FROM ALL_TAB_COLUMNS t where t.TABLE_NAME=?", tableName);
			break;

		default:
			throw new RuntimeException(String.format(
					"You may not support the current connection database:%s type, but you can use the createSQL interface to continue the operation.",
					productType));
		}

		HashSet<String> columns = new HashSet<>();

		while (rs.next()) {

			if (productType.equals(ProductType.SQLITE))
				columns.add(rs.getString(2));
			else
				columns.add(rs.getString(1));
		}

		if (columns.isEmpty())
			throw new CdmQueryException(String.format(
					"Table or view has no fields or insufficient current user permissions,schema:%s tableName:%s",
					schema, tableName));

		return columns;
	}

	/**
	 * 
	 * @param productType
	 * @param sql
	 * @param limit
	 * @return
	 */
	public static boolean processLimit(ProductType productType, StringBuilder sql, Integer[] limit) {

		if (productType == null || sql == null || limit == null || limit[0] == null)
			return false;

		switch (productType) {
		case MYSQL:
			proceessCommonLimit(sql, limit);
			return true;

		case SQLITE:
			proceessCommonLimit(sql, limit);
			return true;

		case MICROSOFT_SQL_SERVER:
			proceessSqlServerLimit(sql, limit);
			return true;

		case ORACLE:
			proceessOracleLimit(sql, limit);
			return true;

		default:
			return false;
		}
	}

	private static void proceessOracleLimit(StringBuilder sql, Integer[] limit) {

		String[] split = sql.toString().split("from");
		String preffix = "SELECT * FROM (" + split[0] + ",rownum rn FROM " + split[1] + ") v WHERE v.rn";
		if (limit.length == 1)
			preffix += "<=" + limit[0];

		if (limit.length == 2)
			preffix += ">" + limit[0] + " AND v.rn<=" + (limit[0] + limit[1]);

		sql.setLength(0);
		sql.append(preffix);

	}

	private static void proceessCommonLimit(StringBuilder sql, Integer[] limit) {

		sql.append(" LIMIT " + limit[0]);
		if (limit.length > 1 && limit[1] != null && limit[1] > 0)
			sql.append("," + limit[1]);

	}

	private static void proceessSqlServerLimit(StringBuilder sql, Integer[] limit) {

		sql.replace(0, 6, "SELECT TOP " + limit[0]).toString();
	}

}
