/*
 * Copyright (C) 2006 Johan Maasing johan at zoom.nu Licensed under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0 Unless required by applicable law
 * or agreed to in writing, software distributed under the License is
 * distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package nu.zoom.swing.desktop.component.stringmenu;

import java.io.Serializable;

import javax.swing.Icon;
import javax.swing.JMenu;

import nu.zoom.swing.desktop.worker.EventQueuePolicy;
import nu.zoom.swing.desktop.worker.Policy;

/**
 * A managed menu of items. The menu saves the items (but not the listeners) to
 * the desktop preferences when the workbench quits. The menu restores itself
 * when the getJMenu is first called. When a menu item is selected by the user
 * all registered listeners will be called.
 */
public interface StringMenu<T extends Comparable<T> & Serializable> {
	/**
	 * Add a listener for selection events.
	 * 
	 * @param listener
	 */
	public void addListener(StringMenuListener<T> listener);

	/**
	 * Remove a listener. Idempotent, i.e. you may remove the same listener
	 * several times or remove null to no effect.
	 * 
	 * @param listener
	 */
	public void removeListener(StringMenuListener<T> listener);

	/**
	 * Add an item to the beginning of the menu. If the same item is added
	 * several times only one item will be created on the menu.
	 * 
	 * @param menuItem
	 */
	@EventQueuePolicy(Policy.EVENT_QUEUE)
	public void addItem(StringMenuItem<T> menuItem);

	/**
	 * Remove a named item. If the item is not on the menu this does nothing.
	 * 
	 * @param menuItem
	 */
	@EventQueuePolicy(Policy.EVENT_QUEUE)
	public void removeItem(StringMenuItem<T> menuItem);

	/**
	 * Get a menu of the items. The menu will be updated when items are added
	 * and removed. Will always return the same instance of the menu.
	 * 
	 * @param menuName
	 *            The name of the menu.
	 * @param icon
	 *            The icon to set for the menu, if null no icon will be used.
	 * @param clearName
	 *            The name of the "clear all"-action at the top of the menu. If
	 *            null the clear action will not be created.
	 * @param clearTooltip
	 *            The tooltip to set for the clear action, if null no tooltip
	 *            will be set.
	 * @return Return the instance of the JMenu this class wraps. Possibly empty
	 *         menu, never null.
	 */
	@EventQueuePolicy(Policy.EVENT_QUEUE)
	JMenu getJMenu(final String menuName, final Icon icon,
			final String clearName, final String clearTooltip);

	/**
	 * Get the number of items on the menu.
	 * 
	 * @return The number of items on the menu.
	 */
	public int getNumberOfItems();

	/**
	 * Remove all items.
	 * 
	 */
	@EventQueuePolicy(Policy.EVENT_QUEUE)
	public void clear();

}
