/*
 * Copyright (C) 2005 Johan Maasing johan at zoom.nu Licensed under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0 Unless required by applicable law
 * or agreed to in writing, software distributed under the License is
 * distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package nu.zoom.ldap.eon.directory.tree;

import java.awt.EventQueue;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Iterator;

import javax.naming.Name;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;
import javax.naming.ldap.InitialLdapContext;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.MutableTreeNode;
import javax.swing.tree.TreeNode;

import nu.zoom.ldap.eon.operation.Operation;
import nu.zoom.ldap.eon.operation.OperationManager;
import nu.zoom.swing.desktop.Workbench;

import org.ops4j.gaderian.Messages;

/**
 * @author $Author: johan $
 * @version $Revision: 1.10 $
 */
public class DirectoryTreeNode implements MutableTreeNode {

    private DirectoryTreeObject directoryTreeObject;
    private ArrayList<DirectoryTreeNode> childNodes = new ArrayList<DirectoryTreeNode>();
    private DirectoryTreeNode parent;
    private boolean staleChildCache = true;
    private DirectoryTree tree;
    private DefaultTreeModel treeModel;
    private Messages messages;
    private OperationManager manager;
    private InitialLdapContext iCtx;
    private Workbench workbench;
    private String presentationName = null;

    DirectoryTreeNode(DirectoryTree tree, Workbench workbench,
            Messages messages, OperationManager manager,
            DefaultTreeModel treeModel, InitialLdapContext iCtx,
            DirectoryTreeNode parent, String rdn) throws NamingException {
        super();
        this.parent = parent;
        this.treeModel = treeModel;
        this.messages = messages;
        this.manager = manager;
        this.iCtx = iCtx;
        this.tree = tree;
        this.workbench = workbench;

        if (rdn.indexOf('/') != -1) {
            rdn = rdn.replaceAll("\"", "").replaceAll("/", "\\/");
        }
        String dn;
        if ((parent != null) && (parent.getDn() != null)) {
            Name thisName = ((Name) parent.getDn().clone()).add(rdn);
            dn = thisName.toString();
        } else {
            // root node or child of root node
            dn = rdn;
        }
        Name nodeName = iCtx.getNameParser("").parse(dn);
        directoryTreeObject = new DirectoryTreeObjectImpl(nodeName, tree);
    }

    DirectoryTreeObject getObject() {
        return directoryTreeObject;
    }

    /*
     * @see javax.swing.tree.TreeNode#getChildAt(int)
     */
    @Override
    public TreeNode getChildAt(int childIndex) {
        if (staleChildCache) {
            listChildren();
        }
        return (TreeNode) childNodes.get(childIndex);
    }

    /*
     * @see javax.swing.tree.TreeNode#getChildCount()
     */
    @Override
    public int getChildCount() {
        if (staleChildCache) {
            listChildren();
        }
        return childNodes.size();
    }

    /*
     * @see javax.swing.tree.TreeNode#getParent()
     */
    @Override
    public TreeNode getParent() {
        return parent;
    }

    /*
     * @see javax.swing.tree.TreeNode#getIndex(javax.swing.tree.TreeNode)
     */
    @Override
    public int getIndex(TreeNode node) {
        return childNodes.indexOf(node);
    }

    /*
     * @see javax.swing.tree.TreeNode#getAllowsChildren()
     */
    @Override
    public boolean getAllowsChildren() {
        return true;
    }

    /*
     * @see javax.swing.tree.TreeNode#isLeaf()
     */
    @Override
    public boolean isLeaf() {
        if (staleChildCache) {
            return false;
        }

        return (childNodes.size() == 0);
    }

    /*
     * @see javax.swing.tree.TreeNode#children()
     */
    @Override
    public Enumeration children() {
        if (staleChildCache) {
            listChildren();
        }
        return new IteratorEnumerator(childNodes.iterator());
    }

    void clearChildCache() {
        staleChildCache = true;
        childNodes.clear();
    }

    private synchronized void listChildren() {
        manager.runOperation(new Operation() {

            @Override
            public void execute() throws NamingException {
                SearchControls ctrls = new SearchControls();
                ctrls.setSearchScope(SearchControls.ONELEVEL_SCOPE);
                ctrls.setReturningAttributes(new String[0]);
                ctrls.setReturningObjFlag(false);
                NamingEnumeration resultSet = null;

                try {
                    Name searchRoot = directoryTreeObject.getDistinguishedName();
                    final ArrayList<String> childNames = new ArrayList<String>();
                    try {
                        resultSet = iCtx.search(searchRoot, "objectClass=*", ctrls);
                        while (resultSet.hasMore()) {
                            SearchResult result = (SearchResult) resultSet.next();
                            childNames.add(result.getName());
                        }
                    } catch (IndexOutOfBoundsException e) {
                        ////// Nasty hack for JNDI weirdness
                        // When searching in the rootDSE of AD (and it seems for
                        // Siemens DirX also) the JNDI provider
                        // throws an index out of bounds.
                    }
                    result(childNames);
                } finally {
                    if (resultSet != null) {
                        try {
                            resultSet.close();
                        } catch (NamingException e) {
                        }
                    }
                }
            }
        });
    } /*
     * @see nu.zoom.ldap.browser.common.NamingEnumerationResultListener#result(javax.naming.NamingEnumeration)
     */


    public synchronized void result(final ArrayList<String> childNames)
            throws NamingException {
        childNodes.clear();
        for (int i = 0; i < childNames.size(); i++) {
            childNodes.add(new DirectoryTreeNode(tree, workbench, messages,
                    manager, treeModel, iCtx, this, (String) childNames.get(i)));
        }
        staleChildCache = false;
        EventQueue.invokeLater(new Runnable() {

            public void run() {
                treeModel.nodeStructureChanged(DirectoryTreeNode.this);
            }
        });

        notifyAll();
    }

    /**
     * Get the distinguished LDAP name for this tree node.
     *
     * @return The LDAP name of this node.
     */
    Name getDn() {
        return directoryTreeObject.getDistinguishedName();
    }

    /**
     * Get the name relative to the parent of this node.
     */
    Name getRdn() {
        Name dn = directoryTreeObject.getDistinguishedName();
        Name rdn = dn.getSuffix(dn.size() - 1);
        return rdn;
    }

    synchronized DirectoryTreeNode getChildWithRDN(String rdn) {
        if (staleChildCache) {
            listChildren();
            try {
                wait();
            } catch (InterruptedException e) {
            }
        }
        DirectoryTreeNode result = null;
        for (int i = 0; i < childNodes.size(); i++) {
            DirectoryTreeNode child = (DirectoryTreeNode) childNodes.get(i);
            if (child.getRdn().toString().equals(rdn)) {
                result = child;
                break;
            }
        }
        return result;
    }

    public class IteratorEnumerator implements Enumeration {

        private Iterator iterator;

        public IteratorEnumerator(Iterator iterator) {
            this.iterator = iterator;
        }

        /*
         * @see java.util.Enumeration#hasMoreElements()
         */
        @Override
        public boolean hasMoreElements() {
            return iterator.hasNext();
        }

        /*
         * @see java.util.Enumeration#nextElement()
         */
        @Override
        public Object nextElement() {
            return iterator.next();
        }
    }

    /*
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        if (presentationName == null) {
            if (parent == null) {
                try {
                    presentationName = iCtx.getNameInNamespace();
                } catch (NamingException e) {
                    presentationName = getDn().toString();
                }
                if (presentationName.length() < 1) {
                    presentationName = "[Root]";
                }
            } else {
                return getRdn().toString();
            }
        }
        return presentationName;
    }

    /*
     * (non-Javadoc)
     *
     * @see javax.swing.tree.MutableTreeNode#insert(javax.swing.tree.MutableTreeNode,
     *      int)
     */
    @Override
    public void insert(MutableTreeNode child, int index) {
        if (child instanceof DirectoryTreeNode) {
            childNodes.add(index, (DirectoryTreeNode) child);
        } else {
            throw new IllegalArgumentException(
                    "Will only add DirectoryTreeNode to the tree");
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see javax.swing.tree.MutableTreeNode#remove(int)
     */
    @Override
    public void remove(int index) {
        childNodes.remove(index);
    }

    /*
     * (non-Javadoc)
     *
     * @see javax.swing.tree.MutableTreeNode#remove(javax.swing.tree.MutableTreeNode)
     */
    @Override
    public void remove(MutableTreeNode node) {
        childNodes.remove(node);
    }

    /*
     * (non-Javadoc)
     *
     * @see javax.swing.tree.MutableTreeNode#removeFromParent()
     */
    @Override
    public void removeFromParent() {
        parent.remove(this);
    }

    /*
     * (non-Javadoc)
     *
     * @see javax.swing.tree.MutableTreeNode#setParent(javax.swing.tree.MutableTreeNode)
     */
    @Override
    public void setParent(MutableTreeNode newParent) {
        if (!(newParent instanceof DirectoryTreeNode)) {
            throw new IllegalArgumentException(
                    "Will only parent to LdapTreeNode");
        }
        // String dn = getRdn();
        // // if ((parent != null) && (parent.getDn() != null)
        // // && (parent.getDn().length() > 0)) {
        // // dn = getRdn() + "," + parent.getDn();
        // // } else {
        // // // root node or child of root node
        // // dn = getRdn();
        // // }
        // try {
        // directoryTreeObject = new DirectoryTreeObjectImpl(iCtx
        // .getNameParser("").parse(dn), tree);
        // } catch (NamingException exc) {
        // workbench.getErrorReporter().reportError(exc);
        // }
        throw new IllegalStateException("Not implemented yet");
    }

    /*
     * (non-Javadoc)
     *
     * @see javax.swing.tree.MutableTreeNode#setUserObject(java.lang.Object)
     */
    @Override
    public void setUserObject(Object object) {
        throw new IllegalArgumentException("LdapTreeNodes have no user object");
    }
}
