/*
 * Copyright (C) 2005 Johan Maasing johan at zoom.nu Licensed under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0 Unless required by applicable law
 * or agreed to in writing, software distributed under the License is
 * distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package nu.zoom.ldap.eon.directory.tree;

import java.awt.BorderLayout;
import java.awt.EventQueue;
import java.awt.event.ActionEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.List;

import javax.naming.Name;
import javax.naming.NamingException;
import javax.naming.ldap.InitialLdapContext;
import javax.swing.AbstractAction;
import javax.swing.JComponent;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JTree;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreePath;

import nu.zoom.ldap.eon.connection.ConnectionInformation;
import nu.zoom.ldap.eon.directory.event.DirectoryEventListener;
import nu.zoom.ldap.eon.event.ActionUtils;
import nu.zoom.ldap.eon.operation.OperationManager;
import nu.zoom.swing.desktop.Workbench;

import org.ops4j.gaderian.Messages;

/**
 * @version $Revision: 1.14 $
 * @author $Author: johan $
 */
public class DirectoryTree extends JComponent implements TreeSelectionListener,
        DirectoryEventListener {

    // Describes the current connection so users know which tree this is.
    private ConnectionInformation connectionDescription;
    private DefaultTreeModel treeModel;
    private Workbench workbench;
    private Messages messages;
    private InitialLdapContext iCtx;
    private OperationManager manager;
    private List<DirectoryTreePopUpItem> popupItems;
    private List<DirectoryTreeSelectionListener> selectionChangeListeners = new ArrayList<DirectoryTreeSelectionListener>();
    private JPopupMenu popupMenu;
    private JTree tree;

    public DirectoryTree(OperationManager manager, Workbench workbench,
            Messages messages, InitialLdapContext iCtx,
            ConnectionInformation connectionDescription,
            List<DirectoryTreePopUpItem> popupItems) {
        super();
        this.iCtx = iCtx;
        this.messages = messages;
        this.workbench = workbench;
        this.manager = manager;
        this.connectionDescription = connectionDescription;
        this.popupItems = popupItems;
        if (EventQueue.isDispatchThread()) {
            buildGUI();
        } else {
            EventQueue.invokeLater(new Runnable() {

                public void run() {
                    buildGUI();
                }
            });
        }
    }

    /**
     * Tries to show the given distinguished name in the tree. The tree will
     * expand to show the given name. This will only operate on a cached copy of
     * the directory tree.
     *
     * @param dn
     *            The name to show.
     */
    public void showDN(Name dn) {

        if ((treeModel == null) || (treeModel.getRoot() == null)) {
            return;
        }
        DirectoryTreeNode currentLdapTreeNode = (DirectoryTreeNode) treeModel.getRoot();
        TreePath path = new TreePath(currentLdapTreeNode);
        for (int i = 0; i < dn.size(); i++) {
            String namePart = dn.get(i);
            DirectoryTreeNode child = currentLdapTreeNode.getChildWithRDN(namePart);
            if (child != null) {
                path = path.pathByAddingChild(child);
                currentLdapTreeNode = child;
            } else {
                break;
            }
        }
        tree.expandPath(path);
        tree.setSelectionPath(path);
        tree.scrollPathToVisible(path);
        tree.validate();
    }

    /**
     * Refresh the cached copy of the children of the named node.
     *
     * @param dn
     *            The node that needs refreshing.
     */
    public void refresh(Name dn) {

        if ((treeModel == null) || (treeModel.getRoot() == null)) {
            return;
        }
        DirectoryTreeNode currentLdapTreeNode = (DirectoryTreeNode) treeModel.getRoot();

        for (int i = 0; i < dn.size(); i++) {
            String namePart = dn.get(i);
            DirectoryTreeNode child = currentLdapTreeNode.getChildWithRDN(namePart);
            if (child == null) {
                break;
            } else {
                currentLdapTreeNode = child;
            }
        }
        currentLdapTreeNode.clearChildCache();
    }

    public synchronized void addDirectoryTreeSelectionListener(
            DirectoryTreeSelectionListener listener) {
        if (listener != null) {
            selectionChangeListeners.add(listener);
        }
    }

    public synchronized void removeDirectoryTreeSelectionListener(
            DirectoryTreeSelectionListener listener) {
        selectionChangeListeners.remove(listener);
    }

    private void buildGUI() {
        // Build Tree
        treeModel = new DefaultTreeModel(null);
        try {
            DirectoryTreeNode rootNode = new DirectoryTreeNode(this, workbench,
                    messages, manager, treeModel, iCtx, null, "");
            treeModel.setRoot(rootNode);
        } catch (NamingException exc) {
            workbench.getErrorReporter().reportError(
                    messages.getMessage("error.tree.rootnode"), exc);
        }
        tree = new JTree(treeModel);

        // Build Tree popup menu
        popupMenu = new JPopupMenu(messages.getMessage("tree.popupmenu"));
        for (DirectoryTreePopUpItem item : popupItems) {
            popupMenu.add(new PopupAction(item));
        }
        tree.addMouseListener(new PopupListener());
        tree.addTreeSelectionListener(this);

        // Layout
        setLayout(new BorderLayout());
        add(new JScrollPane(tree), BorderLayout.CENTER);
    }

    /*
     * (non-Javadoc)
     *
     * @see javax.swing.event.TreeSelectionListener#valueChanged(javax.swing.event.TreeSelectionEvent)
     */
    public synchronized void valueChanged(TreeSelectionEvent e) {
        for (int i = 0; i < selectionChangeListeners.size(); i++) {
            selectionChangeListeners.get(i).selectionChanged(this, iCtx);
        }
    }

    /**
     * Returns the name of the first selected node.
     *
     * @see JTree#getSelectionPath()
     * @return The name of the first selected node, null if nothing selected.
     * @throws NamingException
     */
    public DirectoryTreeObject getSelectedNode() throws NamingException {
        TreePath selectedPath = tree.getSelectionPath();
        if (selectedPath != null) {
            DirectoryTreeNode selectedNode = (DirectoryTreeNode) tree.getSelectionPath().getLastPathComponent();
            return selectedNode.getObject();
        } else {
            return null;
        }
    }

    /**
     * Get the selected nodes in the tree.
     *
     * @see JTree#getSelectionPaths()
     * @return The names of the selected objects, null if nothing selected.
     * @throws NamingException
     */
    public DirectoryTreeObject[] getSelectedNodes() throws NamingException {
        TreePath[] selectedPaths = tree.getSelectionPaths();
        if (selectedPaths != null) {
            DirectoryTreeObject[] selectedNames = new DirectoryTreeObject[selectedPaths.length];
            for (int i = 0; i < selectedPaths.length; i++) {
                DirectoryTreeNode selectedNode = (DirectoryTreeNode) selectedPaths[i].getLastPathComponent();
                selectedNames[i] = selectedNode.getObject();
            }
            return selectedNames;
        } else {
            return null;
        }
    }

    /**
     * Describes the current connection so users know which tree this is.
     *
     * @return Returns the connectionDescription.
     */
    public ConnectionInformation getConnectionInformation() {
        return connectionDescription;
    }

    /**
     * Get the connection that this tree uses.
     *
     * @return The connection the tree uses.
     */
    public InitialLdapContext getConnection() {
        return iCtx;
    }

    /*
     * (non-Javadoc)
     *
     * @see nu.zoom.ldap.eon.directory.event.DirectoryEventListener#structureChanged(javax.naming.Name)
     */
    public void structureChanged(Name name) {
        refresh(name);
        tree.invalidate();
        repaint();
    }

    /*
     * (non-Javadoc)
     *
     * @see nu.zoom.ldap.eon.directory.event.DirectoryEventListener#attributesChanged(javax.naming.Name)
     */
    public void attributesChanged(Name name) {
        // Nothing to do for the tree.
    }

    class PopupListener extends MouseAdapter {

        public void mousePressed(MouseEvent e) {
            maybeShowPopup(e);
        }

        public void mouseReleased(MouseEvent e) {
            maybeShowPopup(e);
        }

        private void maybeShowPopup(MouseEvent e) {
            if (e.isPopupTrigger()) {
                int row = tree.getRowForLocation(e.getX(), e.getY());
                if (tree.getSelectionCount() < 2) {
                    tree.setSelectionRow(row);
                }
                popupMenu.show(e.getComponent(), e.getX(), e.getY());
            }
        }
    }

    class PopupAction extends AbstractAction {

        private DirectoryTreePopUpItem item;

        PopupAction(DirectoryTreePopUpItem item) {
            super();
            this.item = item;
            ActionUtils.setActionValues(this, item, tree);
        }

        /*
         * (non-Javadoc)
         *
         * @see nu.zoom.swing.desktop.component.WorkbenchMessageAction#actionPerformed(java.awt.event.ActionEvent)
         */
        public void actionPerformed(ActionEvent e) {
            item.activate(DirectoryTree.this);
        }
    }
}
