/*
 * Copyright (C) 2005 Johan Maasing johan at zoom.nu Licensed under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0 Unless required by applicable law
 * or agreed to in writing, software distributed under the License is
 * distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package nu.zoom.ldap.eon.directory.attributes;

import java.awt.BorderLayout;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.util.Iterator;
import java.util.Vector;

import javax.naming.Name;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.Attribute;
import javax.naming.directory.BasicAttribute;
import javax.naming.directory.DirContext;
import javax.naming.directory.ModificationItem;
import javax.naming.ldap.InitialLdapContext;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.table.DefaultTableModel;
import javax.swing.table.TableModel;

import nu.zoom.ldap.eon.connection.ConnectionInformation;
import nu.zoom.ldap.eon.directory.event.DirectoryEventListenerRegistry;
import nu.zoom.ldap.eon.operation.Operation;
import nu.zoom.ldap.eon.operation.OperationManager;
import nu.zoom.swing.desktop.Workbench;
import nu.zoom.swing.desktop.common.action.WorkbenchAction;

import org.ops4j.gaderian.Messages;

/**
 * @author $Author: johan $
 * @version $Revision: 1.6 $
 */
public class StringEditor implements AttributeEditor
{

	private JDialog dlg;
	private String attributeID;
	private JTable valueTable;
	private Workbench workbench;
	private Messages messages;
	private OperationManager operationManager;
	private InitialLdapContext iCtx;
	private Name name;
	private DirectoryEventListenerRegistry eventListeners;
	private ConnectionInformation connectionInformation;

	public StringEditor(Workbench workbench, Messages messages,
			OperationManager operationManager,
			DirectoryEventListenerRegistry eventListeners) {
		super();
		this.messages = messages;
		this.workbench = workbench;
		this.operationManager = operationManager;
		this.eventListeners = eventListeners;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see nu.zoom.ldap.eon.directory.attributes.AttributeEditor#edit(javax.naming.ldap.InitialLdapContext,
	 *      javax.naming.Name, javax.naming.directory.Attribute)
	 */
	@SuppressWarnings("unchecked")
	public void edit(
			ConnectionInformation connectionInformation,
			InitialLdapContext iCtx,
			Name name,
			Attribute attribute) {
		this.attributeID = attribute.getID();
		this.iCtx = iCtx;
		this.name = name;
		this.connectionInformation = connectionInformation;
		Vector values = new Vector();
		try {
			NamingEnumeration valueEnum = attribute.getAll();
			while (valueEnum.hasMore()) {
				Vector rowData = new Vector();
				rowData.add(valueEnum.next());
				values.add(rowData);
			}
			JDialog dlg = buildGUI(values);
			dlg.setVisible(true);
		} catch (NamingException e) {
			workbench.getErrorReporter().reportError(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see nu.zoom.ldap.eon.directory.attributes.AttributeEditor#create(javax.naming.ldap.InitialLdapContext,
	 *      javax.naming.Name, java.lang.String)
	 */
	public void create(
			ConnectionInformation connectionInformation,
			InitialLdapContext iCtx,
			Name name,
			String newAttributeID) {
		this.attributeID = newAttributeID;
		this.iCtx = iCtx;
		this.name = name;
		this.connectionInformation = connectionInformation;
		JDialog dlg = buildGUI(null);
		dlg.setVisible(true);
	}

	private JDialog buildGUI(Vector editValues) {
		dlg = new JDialog(workbench.getDialogOwner(), messages.format(
				"attributes.stringeditor.title",
				attributeID), true);
		dlg.setLocationRelativeTo(workbench.getDialogOwner());
		dlg.setDefaultCloseOperation(JDialog.DISPOSE_ON_CLOSE);
		Vector<String> header = new Vector<String>();
		header.add(messages.getMessage("attributes.table.header.value"));
		if (editValues == null) {
			editValues = new Vector();
		}
		valueTable = new JTable(editValues, header);
		valueTable.setEnabled(true);
		valueTable.setShowGrid(true);
		JScrollPane tableScroller = new JScrollPane(valueTable);

		JButton okButton = new JButton(messages
				.getMessage("attributes.stringeditor.action.ok"));
		okButton.addActionListener(new ActionListener() {

			public void actionPerformed(ActionEvent e) {
				saveAttributeReplaceValues();
				dlg.dispose();

			}
		});
		JButton cancelButton = new JButton(messages
				.getMessage("attributes.stringeditor.action.cancel"));
		cancelButton.addActionListener(new ActionListener() {

			public void actionPerformed(ActionEvent e) {
				workbench
						.setStatusbarMessage(messages
								.getMessage("attributes.stringeditor.action.cancelled"));
				dlg.dispose();
			}
		});

		AddValueAction addAction = new AddValueAction();
		JButton addValueButton = new JButton(addAction);
		DeleteValueAction deleteAction = new DeleteValueAction();
		JButton deleteValueButton = new JButton(deleteAction);

		JPanel buttonPanel = new JPanel(new FlowLayout());
		buttonPanel.add(addValueButton);
		buttonPanel.add(deleteValueButton);
		buttonPanel.add(okButton);
		buttonPanel.add(cancelButton);

		dlg.getContentPane().setLayout(new BorderLayout());
		dlg.getContentPane().add(tableScroller, BorderLayout.CENTER);
		dlg.getContentPane().add(buttonPanel, BorderLayout.SOUTH);

		dlg.pack();

		return dlg;
	}

	/**
	 * Replaces the values in the directory with the ones from the table model.
	 * Creates the attribute if it does not exist. Removes the attribute if no
	 * values are specified.
	 * 
	 * @see DirContext#REPLACE_ATTRIBUTE
	 * @return true If the attribute could be saved, false if errors occured.
	 */
	private void saveAttributeReplaceValues() {
		TableModel model = valueTable.getModel();
		if (model instanceof DefaultTableModel) {
			Vector rows = ((DefaultTableModel) model).getDataVector();
			if (rows != null) {
				final BasicAttribute attribute = new BasicAttribute(attributeID);
				for (Iterator valueIterator = rows.iterator(); valueIterator
						.hasNext();) {
					Vector columns = (Vector) valueIterator.next();
					if ((columns != null) && (columns.size() > 0)) {
						String value = (String) columns.get(0);
						attribute.add(value);
					}
				}
				operationManager.runOperation(new Operation() {

					public void execute() throws NamingException {
						ModificationItem moditem = new ModificationItem(
								DirContext.REPLACE_ATTRIBUTE,
								attribute);
						iCtx.modifyAttributes(
								name,
								new ModificationItem[] { moditem });
						eventListeners.fireAttributesChanged(
								connectionInformation.getGUID(),
								name);

					}
				});
			}
		}
	}

	class AddValueAction extends WorkbenchAction
	{

		AddValueAction() {
			setName(messages.getMessage("attributes.stringeditor.action.add"));
			setAcceleratorKey(KeyEvent.VK_A, KeyEvent.CTRL_DOWN_MASK);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see nu.zoom.swing.desktop.component.WorkbenchAction#actionPerformed(java.awt.event.ActionEvent)
		 */
		@Override
		public void actionPerformed(ActionEvent e) {
			String newValue = JOptionPane.showInputDialog(dlg, messages
					.getMessage("attributes.stringeditor.action.add"));
			if ((newValue != null) && (newValue.length() > 0)) {
				TableModel model = valueTable.getModel();
				if (model instanceof DefaultTableModel) {
					Vector<String> row = new Vector<String>();
					row.add(newValue);
					((DefaultTableModel) model).addRow(row);
				}
			}
		}
	}

	class DeleteValueAction extends WorkbenchAction
	{

		DeleteValueAction() {
			setName(messages
					.getMessage("attributes.stringeditor.action.delete"));
			setAcceleratorKey(KeyEvent.VK_A, KeyEvent.CTRL_DOWN_MASK);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see nu.zoom.swing.desktop.component.WorkbenchAction#actionPerformed(java.awt.event.ActionEvent)
		 */
		@Override
		public void actionPerformed(ActionEvent e) {
			int selectedRow = valueTable.getSelectedRow();
			if (selectedRow > -1)
			{
				TableModel model = valueTable.getModel();
				if (model instanceof DefaultTableModel)
				{
					((DefaultTableModel) model).removeRow(selectedRow);
				}
			}
		}
	}
}
