/*
 * Copyright (C) 2004 Johan Maasing johan at zoom.nu
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package nu.zoom.ldap.eon.directory.attributes;

import java.util.ArrayList;
import java.util.Collections;

import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.Attribute;
import javax.naming.directory.Attributes;
import javax.swing.table.AbstractTableModel;

/**
 * Table model to display Attributes.
 * 
 * @author $Author: johan $
 * @version $Revision: 1.2 $
 */
public class AttributesTableModel extends AbstractTableModel
{

	private ArrayList<AttributeTableRow> attributeRows = new ArrayList<AttributeTableRow>();
	private String[] headernames = new String[2];

	/**
	 * Create a new attributes table model
	 * 
	 * @param nameHeader
	 *            The column header for the attribute name column.
	 * @param valueHeader
	 *            The column header for the attribute value column.
	 */
	public AttributesTableModel(String nameHeader, String valueHeader) {
		super();
		headernames[0] = nameHeader;
		headernames[1] = valueHeader;
	}

	/**
	 * Set the model to show the given attributes.
	 * 
	 * @param attrs
	 *            The attributes collection to display in the table.
	 * @throws NamingException
	 */
	public void resetModel(Attributes attrs) throws NamingException {
		if (attrs == null)
			throw new IllegalArgumentException("Attributes can not be null");

		attributeRows.clear();

		NamingEnumeration attributeEnumeration = attrs.getAll();
		while (attributeEnumeration.hasMore()) {
			Attribute attr = (Attribute) attributeEnumeration.next();
			String name = attr.getID();
			NamingEnumeration attributeValueEnumeration = attr.getAll();
			while (attributeValueEnumeration.hasMore()) {
				Object attrValue = attributeValueEnumeration.next();
				String presentationValue;
				if (attrValue instanceof String) {
					presentationValue = (String) attrValue;
				} else {
					presentationValue = "...";
				}

				AttributeTableRow row = new AttributeTableRow();
				row.setAttribute(attr);
				row.setId(name);
				row.setValue(presentationValue);
				attributeRows.add(row);
			}
		}
		Collections.<AttributeTableRow>sort(attributeRows);
		fireTableDataChanged();
	}

	/*
	 * @see javax.swing.table.TableModel#getRowCount()
	 */
	public int getRowCount() {
		return attributeRows.size();
	}

	/*
	 * @see javax.swing.table.TableModel#getColumnCount()
	 */
	public int getColumnCount() {
		return headernames.length;
	}

	/*
	 * @see javax.swing.table.TableModel#getValueAt(int, int)
	 */
	public Object getValueAt(int rowIndex, int columnIndex) {
		switch (columnIndex) {
			case 0:
				return ((AttributeTableRow) attributeRows.get(rowIndex))
						.getId();
			case 1:
				return ((AttributeTableRow) attributeRows.get(rowIndex))
						.getValue();
			default:
				return null;
		}
	}

	/*
	 * @see javax.swing.table.TableModel#getColumnName(int)
	 */
	public String getColumnName(int column) {
		if ((column >= 0) && (column < headernames.length)) {
			return headernames[column];
		} else {
			return super.getColumnName(column);
		}
	}

	public Attribute getAttributeAt(int row) {
		return ((AttributeTableRow) attributeRows.get(row)).getAttribute();
	}

}
