/*
 * Copyright (C) 2005 Johan Maasing johan at zoom.nu Licensed under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0 Unless required by applicable law
 * or agreed to in writing, software distributed under the License is
 * distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package nu.zoom.ldap.eon.directory.attributes;

import java.awt.BorderLayout;
import java.awt.EventQueue;
import java.awt.event.ActionEvent;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;

import javax.naming.Name;
import javax.naming.NamingException;
import javax.naming.directory.Attribute;
import javax.naming.directory.Attributes;
import javax.naming.ldap.InitialLdapContext;
import javax.swing.Action;
import javax.swing.JComponent;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.table.TableColumnModel;
import javax.swing.table.TableModel;

import nu.zoom.ldap.eon.connection.ConnectionInformation;
import nu.zoom.ldap.eon.directory.event.DirectoryEventListener;
import nu.zoom.swing.desktop.Workbench;
import nu.zoom.swing.desktop.common.action.WorkbenchAction;

import org.ops4j.gaderian.Messages;

/**
 * @version $Revision: 1.8 $
 * @author $Author: johan $
 */
public class AllAttributesComponent extends JComponent implements
		MouseListener, DirectoryEventListener
{

	private Workbench workbench;
	private Messages messages;
	private AttributeEditorFactory editorFactory;
	private Name dn;
	private JTable attributesTable;
	private AttributesTableModel model;
	private InitialLdapContext iCtx;
	private JPopupMenu popupMenu;
	private ConnectionInformation connectionInformation;

	public AllAttributesComponent(Workbench workbench, Messages messages,
			AttributeEditorFactory editorFactory, InitialLdapContext iCtx,
			ConnectionInformation connectionInformation) {
		super();
		this.messages = messages;
		this.workbench = workbench;
		this.editorFactory = editorFactory;
		this.iCtx = iCtx;
		this.connectionInformation = connectionInformation;
		if (EventQueue.isDispatchThread()) {
			buildGUI();
		} else {
			EventQueue.invokeLater(new Runnable() {

				public void run() {
					buildGUI();
				}
			});
		}
	}

	private void buildGUI() {
		model = new AttributesTableModel(messages
				.getMessage("attributes.table.header.id"), messages
				.getMessage("attributes.table.header.value"));
		attributesTable = new JTable(model);
		attributesTable.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
		TableColumnModel columnModel = attributesTable.getColumnModel();
		for (int c = 0; c < columnModel.getColumnCount(); c++) {
			columnModel.getColumn(c).setPreferredWidth(250);
		}
		JScrollPane attributesTableScroller = new JScrollPane(attributesTable);

		setLayout(new BorderLayout());
		add(attributesTableScroller, BorderLayout.CENTER);

		validate();

		popupMenu = new JPopupMenu();

		Action addAttributeAction = new AddAttributeAction();
		popupMenu.add(addAttributeAction);
		//		Action deleteAttributeAction = new DeleteAttributeAction();
		//
		//		popupMenu = new JPopupMenu();
		//		popupMenu.add(addAttributeAction);
		//		popupMenu.add(deleteAttributeAction);
		//		popupMenu.add(new EditAttributeAction());
		//		popupMenu.add(new ShowInLeftTreeAction());
		//		popupMenu.add(new ShowInRightTreeAction());
		attributesTable.addMouseListener(this);
	}

	/**
	 * Show the attributes. This method is delayed to be called in the swing
	 * thread so it is safe to call from other threads. Notice that the
	 * attributes must not be manipulated after the call to this method
	 * (concurrency problems otherwise).
	 * 
	 * @param attributes
	 * @throws NamingException
	 */
	public void show(Name dn, final Attributes attributes)
			throws NamingException {
		this.dn = dn;
		EventQueue.invokeLater(new Runnable() {

			public void run() {
				try {
					model.resetModel(attributes);
				} catch (NamingException e) {
					workbench.getErrorReporter().reportError(
							messages.getMessage("error.attributes.show"),
							e);
				}
			}
		});
	}

	private void editSelectedAttribute() {
		Attribute attr = getSelectedAttribute();
		if (attr != null) {
			try {
				AttributeEditor editor = editorFactory.getEditor(attr);
				editor.edit(connectionInformation, iCtx, dn, attr);
			} catch (NamingException e) {
				workbench.getErrorReporter().reportError(e);
			}
		} else {
			workbench.getErrorReporter().reportError(
					messages.getMessage("attributes.noselected"));
		}
	}

	private Attribute getSelectedAttribute() {
		Attribute attr = null;
		int selectedRow = attributesTable.getSelectedRow();
		if (selectedRow >= 0) {
			TableModel tableModel = attributesTable.getModel();
			if ((tableModel != null)
					&& (tableModel instanceof AttributesTableModel)) {
				attr = ((AttributesTableModel) tableModel)
						.getAttributeAt(selectedRow);
			}
		}
		return attr;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.awt.event.MouseListener#mouseEntered(java.awt.event.MouseEvent)
	 */
	public void mouseEntered(MouseEvent e) {
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.awt.event.MouseListener#mouseExited(java.awt.event.MouseEvent)
	 */
	public void mouseExited(MouseEvent e) {
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.awt.event.MouseListener#mousePressed(java.awt.event.MouseEvent)
	 */
	public void mousePressed(MouseEvent e) {
		maybeShowPopup(e);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.awt.event.MouseListener#mouseReleased(java.awt.event.MouseEvent)
	 */
	public void mouseReleased(MouseEvent e) {
		maybeShowPopup(e);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.awt.event.MouseListener#mouseClicked(java.awt.event.MouseEvent)
	 */
	public void mouseClicked(MouseEvent e) {
		if (e.getButton() == MouseEvent.BUTTON3) {
			int rowAt = attributesTable.rowAtPoint(e.getPoint());
			if (rowAt >= 0) {
				attributesTable.setRowSelectionInterval(rowAt, rowAt);
			}
		}
		if (e.getClickCount() == 2) {
			editSelectedAttribute();
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see nu.zoom.ldap.eon.directory.event.DirectoryEventListener#attributesChanged(javax.naming.Name)
	 */
	public void attributesChanged(Name name) {
		if (name.equals(dn)) {
			try {
				show(dn, iCtx.getAttributes(dn));
			} catch (NamingException e) {
				workbench.getErrorReporter().reportError(e);
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see nu.zoom.ldap.eon.directory.event.DirectoryEventListener#structureChanged(javax.naming.Name)
	 */
	public void structureChanged(Name name) {
	}

	private void maybeShowPopup(MouseEvent e) {
		if (e.isPopupTrigger()) {
			popupMenu.show(e.getComponent(), e.getX(), e.getY());
		}
	}

	class AddAttributeAction extends WorkbenchAction
	{

		AddAttributeAction() {
			setName(messages.getMessage("attributes.popup.add"));
			setToolTip(messages.getMessage("attributes.popup.add.tooltip"));
		}

		@Override
		public void actionPerformed(ActionEvent e) {
			AddAttributeDialog dlg = new AddAttributeDialog(workbench, messages);
			dlg.pack();
			dlg.setLocationRelativeTo(workbench.getDialogOwner());
			dlg.setVisible(true);
			String attributeID = dlg.getText();
			if (attributeID != null) {
				AttributeEditor editor = editorFactory.getEditor(attributeID);
				editor.create(connectionInformation, iCtx, dn, attributeID);
			}
		}
	}
}
