/*
 * Copyright (C) 2005 Johan Maasing johan at zoom.nu Licensed under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0 Unless required by applicable law
 * or agreed to in writing, software distributed under the License is
 * distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package nu.zoom.ldap.eon.connection.password;

import java.awt.EventQueue;
import java.util.Hashtable;

import javax.naming.Context;
import javax.naming.NamingException;
import javax.naming.ldap.InitialLdapContext;

import nu.zoom.swing.desktop.Workbench;
import nu.zoom.swing.desktop.common.BackendException;

import org.ops4j.gaderian.Messages;

/**
 * @author $Author: johan $
 * @version $Revision: 1.4 $
 */
public class UserConnection extends AnonymousConnection {
	static final long serialVersionUID = -7973745137178679985L;

	private String username;

	private transient Workbench workbench = null;

	private transient Messages messages = null;

	public UserConnection() {
		super();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see nu.zoom.ldap.eon.connection.Connection#getConnection()
	 */
	public synchronized InitialLdapContext getConnection()
			throws BackendException {
		if ((workbench == null) || (messages == null)) {
			throw new IllegalStateException(
					UserConnection.class.getName()
							+ " is not initalized properly - workbench or messages are null.");
		}
		final PasswordQueryMutex result = new PasswordQueryMutex();
		if (EventQueue.isDispatchThread()) {
			getPasswordOnEventThread(result);
		} else {
			// Schedule the dialog
			EventQueue.invokeLater(new Runnable() {
				public synchronized void run() {
					getPasswordOnEventThread(result);
					notifyAll();
				}
			});
			// Wait for the dialog to finish on the event queue.
			while (!result.isReady()) {
				try {
					wait(123);
				} catch (InterruptedException exc) {
				}
			}
		}
		if (result.isCloseOK()) {
			Hashtable env = new Hashtable();
			setupEnvironment(env, getUsername(), result.getPassword());
			try {
				InitialLdapContext iCtx = new InitialLdapContext(env, null);
				return iCtx;
			} catch (NamingException e) {
				throw new BackendException(e);
			}
		} else {
			return null;
		}
	}

	@SuppressWarnings("unchecked")
	protected void setupEnvironment(Hashtable env, String username,
			char[] password) {
		setupEnvironment(env);
		env.put(Context.SECURITY_PRINCIPAL, username);
		env.put(Context.SECURITY_CREDENTIALS, password);
	}

	/**
	 * @param messages
	 *            The messages to set.
	 */
	public void setMessages(Messages messages) {
		this.messages = messages;
	}

	/**
	 * @param workbench
	 *            The workbench to set.
	 */
	public void setWorkbench(Workbench workbench) {
		this.workbench = workbench;
	}

	/**
	 * @return Returns the username.
	 */
	public String getUsername() {
		return username;
	}

	/**
	 * @param username
	 *            The username to set.
	 */
	public void setUsername(String username) {
		this.username = username;
	}

	private void getPasswordOnEventThread(final PasswordQueryMutex mutex) {
		PasswordDialog dlg = new PasswordDialog(workbench, messages);
		dlg.setVisible(true);
		mutex.closeOK = dlg.isCloseOK();
		if (mutex.closeOK) {
			mutex.password = dlg.getPassword();
		}
		mutex.setRead();
	}

	class PasswordQueryMutex {
		private boolean ready = false;

		char[] password = null;

		boolean closeOK = false;

		synchronized boolean isReady() {
			return ready;
		}

		synchronized void setRead() {
			ready = true;
		}

		char[] getPassword() {
			return password;
		}

		boolean isCloseOK() {
			return closeOK;
		}
	}

}
