/*
 * Copyright (C) 2005 Johan Maasing johan at zoom.nu Licensed under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0 Unless required by applicable law
 * or agreed to in writing, software distributed under the License is
 * distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package nu.zoom.ldap.eon.connection.password;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Set;

import javax.swing.ImageIcon;

import nu.zoom.ldap.eon.connection.AbstractConnectionFactory;
import nu.zoom.ldap.eon.connection.Connection;
import nu.zoom.ldap.eon.connection.ConnectionEditor;
import nu.zoom.ldap.eon.desktop.ComponentFactory;
import nu.zoom.ldap.eon.operation.OperationManager;
import nu.zoom.swing.desktop.Workbench;
import nu.zoom.swing.desktop.common.BackendException;
import nu.zoom.swing.desktop.preferences.InvalidDataTypeException;
import nu.zoom.swing.desktop.preferences.Preferences;

import org.ops4j.gaderian.Messages;

/**
 * @author $Author: johan $
 * @version $Revision: 1.6 $
 */
public class PasswordConnectionFactory extends AbstractConnectionFactory {
	private static final String PREFS_PREFIX = "PasswordConnectionFactory.";

	private static final String PREFS_CONNECTION_IDS = "PasswordConnectionFactory.Collection";

	private Preferences preferences;

	private Workbench workbench;

	private Messages messages;

	private ComponentFactory componentFactory;

	private OperationManager operationManager;

	public PasswordConnectionFactory(Preferences preferences,
			Workbench workbench, Messages messages,
			ComponentFactory componentFactory, OperationManager operationManager) {
		super();
		this.preferences = preferences;
		this.workbench = workbench;
		this.messages = messages;
		this.componentFactory = componentFactory;
		this.operationManager = operationManager;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see nu.zoom.ldap.eon.connection.ConnectionFactory#listConnections()
	 */
	public Connection[] listConnections() throws BackendException {
		try {
			ArrayList<Connection> connections = new ArrayList<Connection>();
			Set<String> connectionIDs = preferences.getStrings(PREFS_CONNECTION_IDS);
			for (String connectionID : connectionIDs) {
				byte[] serialConnection = preferences.getBytes(PREFS_PREFIX
						+ connectionID);
				if (serialConnection != null) {
					try {
						ObjectInputStream in = new ObjectInputStream(
								new ByteArrayInputStream(serialConnection));
						Connection connection = (Connection) in.readObject();
						connections.add(connection);
						// Connections demanding interaction when connecting
						// Needs to have some values set before thay are
						// operational.
						if (connection instanceof UserConnection) {
							((UserConnection) connection)
									.setWorkbench(workbench);
							((UserConnection) connection).setMessages(messages);
						}
					} catch (Exception exc1) {
						throw new BackendException(exc1);
					}
				} else {
					throw new BackendException(
							"Preferences out of sync, listed connection is missing data.");
				}
			}
			return (Connection[]) connections
					.toArray(new Connection[connections.size()]);
		} catch (InvalidDataTypeException exc) {
			throw new BackendException(
					"Unable to get collection of connection names", exc);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see nu.zoom.ldap.eon.connection.AbstractConnectionFactory#delegateAddConnection(nu.zoom.ldap.eon.connection.Connection)
	 */
	protected void delegateAddConnection(Connection connection)
			throws BackendException {
		try {
			Set<String> connectionIDs = preferences
					.getStrings(PREFS_CONNECTION_IDS);
			HashSet<String> newSet = new HashSet<String>(connectionIDs);
			newSet.add(connection.getGUID().toString());
			preferences.setStrings(PREFS_CONNECTION_IDS, newSet);
			saveConnection(connection);
		} catch (Exception exc) {
			throw new BackendException(exc);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see nu.zoom.ldap.eon.connection.AbstractConnectionFactory#delegateRemoveConnection(nu.zoom.ldap.eon.connection.Connection)
	 */
	protected void delegateRemoveConnection(Connection connection)
			throws BackendException {
		String connectionID = connection.getGUID().toString();
		preferences.setBytes(PREFS_PREFIX + connection.getGUID().toString(),
				null);
		try {
			Set<String> connectionIDs = preferences
					.getStrings(PREFS_CONNECTION_IDS);
			HashSet<String> newSet = new HashSet<String>(connectionIDs);
			newSet.remove(connectionID);
			preferences.setStrings(PREFS_CONNECTION_IDS, newSet);
		} catch (InvalidDataTypeException exc) {
			throw new BackendException(exc);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see nu.zoom.ldap.eon.connection.AbstractConnectionFactory#delegateConnectionChanged(nu.zoom.ldap.eon.connection.Connection)
	 */
	protected void delegateConnectionChanged(Connection connection)
			throws BackendException {
		try {
			saveConnection(connection);
		} catch (IOException exc) {
			throw new BackendException(exc);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see nu.zoom.ldap.eon.connection.ConnectionFactory#getDescription()
	 */
	public String getDescription() {
		return messages.getMessage("connection.password.description");
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see nu.zoom.ldap.eon.connection.ConnectionFactory#getName()
	 */
	public String getName() {
		return messages.getMessage("connection.password.name");
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see nu.zoom.ldap.eon.connection.ConnectionFactory#getIcon()
	 */
	public ImageIcon getIcon() {
		return null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see nu.zoom.ldap.eon.connection.ConnectionFactory#getEditor()
	 */
	public ConnectionEditor getEditor() {
		return new PasswordConnectionEditor(workbench, messages, this,
				componentFactory, operationManager);
	}

	private void saveConnection(Connection connection) throws IOException,
			BackendException {
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		ObjectOutputStream out = new ObjectOutputStream(baos);
		out.writeObject(connection);
		preferences.setBytes(PREFS_PREFIX + connection.getGUID().toString(),
				baos.toByteArray());
	}
}
