/*
 * Copyright (C) 2005 Johan Maasing johan at zoom.nu Licensed under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0 Unless required by applicable law
 * or agreed to in writing, software distributed under the License is
 * distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package nu.zoom.ldap.eon.connection.password;

import java.awt.BorderLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JPasswordField;
import javax.swing.JTextField;

import nu.zoom.ldap.eon.connection.AbstractConnectionFactory;
import nu.zoom.ldap.eon.connection.ConnectOperation;
import nu.zoom.ldap.eon.connection.Connection;
import nu.zoom.ldap.eon.connection.ConnectionEditor;
import nu.zoom.ldap.eon.connection.ConnectionGUID;
import nu.zoom.ldap.eon.desktop.ComponentFactory;
import nu.zoom.ldap.eon.operation.OperationManager;
import nu.zoom.ldap.eon.util.MessagesUtil;
import nu.zoom.swing.desktop.Workbench;
import nu.zoom.swing.desktop.WorkbenchFrame;
import nu.zoom.swing.desktop.common.BackendException;
import nu.zoom.swing.layout.VerticalPanel;

import org.ops4j.gaderian.Messages;

/**
 * Editor for LDAP (not SSL) connections that have username/password for subject
 * and credentials.
 * 
 * @author $Author: johan $
 * @version $Revision: 1.5 $
 */
public class PasswordConnectionEditor extends JComponent implements
		ConnectionEditor {
	private Messages messages;

	private AbstractConnectionFactory factory;

	private Workbench workbench;

	private WorkbenchFrame ownerFrame = null;

	private ComponentFactory componentFactory;

	private OperationManager operationManager;

	private static final int CREDENTIALS_FIELD_COLUMNS = 50;

	private static final int PRINCIPAL_FIELD_COLUMNS = CREDENTIALS_FIELD_COLUMNS;

	private static final int PORT_FIELD_COLUMNS = 5;

	private static final int HOST_FIELD_COLUMNS = CREDENTIALS_FIELD_COLUMNS;

	private static final int NAME_FIELD_COLUMNS = CREDENTIALS_FIELD_COLUMNS;

	private static final int ROOT_CONTEXT_FIELD_COLUMNS = CREDENTIALS_FIELD_COLUMNS;

	private static final int MIN_ALLOWED_POTNUMBER = 1;

	private static final int MAX_ALLOWED_PORTNUMBER = 65535;

	private JTextField nameField;

	private JTextField hostField;

	private JTextField portField;

	private JTextField principalField;

	private JTextField rootContextField;

	private JPasswordField credentialsField;

	private JCheckBox anonymousCheckBox;

	private JCheckBox saveCredentialsBox;

	private boolean editingExistingEntry;

	private JLabel headerLabel;

	private JButton okButton;

	private JButton connectButton;

	private Connection connection = null;

	/**
	 * Construct an editor. The constructor must be called on the AWT dispatch
	 * thread.
	 * 
	 * @param messages
	 *            The message bundle to use.
	 * @param factory
	 *            The factory that produced this connection.
	 */
	public PasswordConnectionEditor(Workbench workbench, Messages messages,
			AbstractConnectionFactory factory,
			ComponentFactory componentFactory, OperationManager operationManager) {
		super();
		this.messages = messages;
		this.factory = factory;
		this.workbench = workbench;
		this.componentFactory = componentFactory;
		this.operationManager = operationManager;
		buildGUI();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see nu.zoom.ldap.eon.connection.ConnectionEditor#createNewConnectionEditor()
	 */
	public JComponent createNewConnectionEditor() {
		initNewConnection();
		return this;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see nu.zoom.ldap.eon.connection.ConnectionEditor#editConnection(nu.zoom.ldap.eon.connection.Connection)
	 */
	public JComponent editConnection(Connection connection) {
		this.connection = connection;
		initEditConnection();
		return this;
	}

	private void initNewConnection() {
		editingExistingEntry = false;
		headerLabel.setText(messages
				.getMessage("connection.password.header.new"));
		hostField.setText("localhost");
		portField.setText("389");
		rootContextField.setText("");
		anonymousCheckBox.setSelected(false);
		saveCredentialsBox.setSelected(false);
		principalField.setText("");
		credentialsField.setText("");
		nameField.setText("");
	}

	private void initEditConnection() {
		editingExistingEntry = true;
		headerLabel.setText(messages
				.getMessage("connection.password.header.edit"));
		// Set common attributes
		hostField.setText(((AnonymousConnection) connection).getHost());
		portField.setText("" + ((AnonymousConnection) connection).getPort());
		nameField.setText(((AnonymousConnection) connection).getDescription());
		rootContextField.setText(((AnonymousConnection) connection)
				.getRootContext());

		// Set checkboxes according to type
		if (connection instanceof UserConnection) {
			anonymousCheckBox.setSelected(false);
			saveCredentialsBox.setSelected(false);
			principalField.setText(((UserConnection) connection).getUsername());
			credentialsField.setText("");
		} else if (connection instanceof AnonymousConnection) {
			anonymousCheckBox.setSelected(true);
			saveCredentialsBox.setSelected(false);
			principalField.setText("");
			credentialsField.setText("");
		} else {
			anonymousCheckBox.setSelected(false);
			saveCredentialsBox.setSelected(false);
			principalField.setText(((UserPasswordConnection) connection)
					.getUsername());
			credentialsField.setText(new String(
					((UserPasswordConnection) connection).getPassword()));
		}

	}

	private void buildGUI() {
		JLabel bannerLabel = new JLabel(MessagesUtil.getIconFromMessages(
				messages, "dialog.banner"));
		headerLabel = new JLabel();
		nameField = new JTextField(NAME_FIELD_COLUMNS);
		hostField = new JTextField(HOST_FIELD_COLUMNS);
		portField = new JTextField(PORT_FIELD_COLUMNS);
		rootContextField = new JTextField(ROOT_CONTEXT_FIELD_COLUMNS);
		credentialsField = new JPasswordField(CREDENTIALS_FIELD_COLUMNS);
		anonymousCheckBox = new JCheckBox("");
		anonymousCheckBox.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				principalField.setEnabled(!anonymousCheckBox.isSelected());
				if (anonymousCheckBox.isSelected()) {
					credentialsField.setEnabled(false);
				} else {
					credentialsField
							.setEnabled(saveCredentialsBox.isSelected());
				}
				saveCredentialsBox.setEnabled(!anonymousCheckBox.isSelected());
			}
		});

		saveCredentialsBox = new JCheckBox(messages
				.getMessage("connection.password.save"));
		saveCredentialsBox.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				credentialsField.setEnabled(saveCredentialsBox.isSelected());
			}
		});
		saveCredentialsBox.setToolTipText(messages
				.getMessage("connection.password.save.tooltip"));
		credentialsField.setEnabled(false);
		principalField = new JTextField(PRINCIPAL_FIELD_COLUMNS);
		credentialsField.setEnabled(false);

		JLabel connectionNameLabel = new JLabel(messages
				.getMessage("connection.password.connectionname"));
		JLabel hostnameLabel = new JLabel(messages
				.getMessage("connection.password.host"));
		JLabel portLabel = new JLabel(messages
				.getMessage("connection.password.portnumber"));
		JLabel rootContextLabel = new JLabel(messages
				.getMessage("connection.password.context"));
		JLabel anonymousLabel = new JLabel(messages
				.getMessage("connection.password.anonymous"));
		JLabel principalLabel = new JLabel(messages
				.getMessage("connection.password.principal"));
		connectButton = new JButton(messages
				.getMessage("connection.password.connect"));
		connectButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				if (validateFields()) {
					createConnection();
					ConnectOperation operation = new ConnectOperation(
							workbench, messages, connection, componentFactory);
					operationManager.runOperation(operation);
				}
			}
		});
		okButton = new JButton(messages.getMessage("connection.password.ok"));
		okButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				if (validateFields()) {
					createConnection();
					if (editingExistingEntry) {
						try {
							factory.connectionChanged(connection);
							headerLabel.setText(messages
									.getMessage("connection.password.saved"));
						} catch (BackendException exc) {
							workbench.getErrorReporter().reportError(exc);
						}
					} else {
						try {
							factory.addConnection(connection);
							headerLabel.setText(messages
									.getMessage("connection.password.created"));
							okButton.setEnabled(false);
							if (ownerFrame != null) {
								ownerFrame.dispose();
							}
						} catch (BackendException exc) {
							workbench.getErrorReporter().reportError(exc);
						}
					}
				}
			}
		});

		JPanel buttonPanel = new JPanel();
		buttonPanel.add(connectButton);
		buttonPanel.add(okButton);

		VerticalPanel mainPanel = new VerticalPanel();
		mainPanel.addRow(bannerLabel, headerLabel);
		mainPanel.addRow(connectionNameLabel, nameField);
		mainPanel.addRow(hostnameLabel, hostField);
		mainPanel.addRow(portLabel, portField);
		mainPanel.addRow(rootContextLabel, rootContextField);
		mainPanel.addRow(anonymousLabel, anonymousCheckBox);
		mainPanel.addRow(principalLabel, principalField);
		mainPanel.addRow(saveCredentialsBox, credentialsField);
		mainPanel.addRow(buttonPanel);
		setLayout(new BorderLayout());
		add(mainPanel, BorderLayout.CENTER);
	}

	private boolean validateFields() {
		String host = hostField.getText();
		if ((host == null) || (host.trim().length() < 1)) {
			complain("connection.password.error.missing.host");
			return false;
		}
		String port = portField.getText();
		if ((port == null) || (port.trim().length() < 1)) {
			complain("connection.password.error.missing.port");
			return false;
		} else {
			try {
				int portNum = Integer.parseInt(port);
				if ((portNum < MIN_ALLOWED_POTNUMBER)
						|| (portNum > MAX_ALLOWED_PORTNUMBER)) {
					complain("connection.password.error.port.invalid");
					return false;
				}
			} catch (NumberFormatException e) {
				complain("connection.password.error.port.invalid");
				return false;
			}
		}
		if (!anonymousCheckBox.isSelected()) {
			String principal = principalField.getText();
			if ((principal == null) || (principal.trim().length() < 1)) {
				complain("connection.password.error.missing.principal");
				return false;
			}
		}
		String name = nameField.getText();
		if ((name == null) || (name.trim().length() < 1)) {
			complain("connection.password.error.missing.name");
			return false;
		}
		return true;
	}

	private void complain(String messageKey) {
		workbench.getErrorReporter().reportError(
				messages.getMessage(messageKey));
	}

	private void createConnection() {
		ConnectionGUID connectionGUID = null;
		if (connection != null) {
			// Modify an existing connection
			connectionGUID = connection.getGUID();
		}
		if (anonymousCheckBox.isSelected()) {
			connection = new AnonymousConnection();
		} else {
			if (saveCredentialsBox.isSelected()) {
				connection = new UserPasswordConnection();
			} else {
				connection = new UserConnection();
			}
		}

		// Restore GUID if we are modifying an existing connection
		if (connectionGUID != null) {
			((AnonymousConnection) connection).setGUID(connectionGUID);
		}

		// Set the values
		if (connection instanceof UserPasswordConnection) {
			((UserPasswordConnection) connection).setPassword(credentialsField
					.getPassword());
		}
		if (connection instanceof UserConnection) {
			((UserConnection) connection).setUsername(principalField.getText()
					.trim());
			((UserConnection) connection).setMessages(messages);
			((UserConnection) connection).setWorkbench(workbench);
		}
		if (connection instanceof AnonymousConnection) {
			((AnonymousConnection) connection).setHost(hostField.getText()
					.trim());
			((AnonymousConnection) connection).setPort(Integer
					.parseInt(portField.getText()));
			((AnonymousConnection) connection).setDescription(nameField
					.getText().trim());
			if (rootContextField.getText().trim().length() < 1) {
				((AnonymousConnection) connection).setRootContext(null);
			} else {
				((AnonymousConnection) connection)
						.setRootContext(rootContextField.getText().trim());
			}

		}
	}

	public void setOwnerFrame(WorkbenchFrame frame) {
		ownerFrame = frame;
	}
}
