/*
 *  Copyright 2009 Johan Maasing johan@zoom.nu.
 * 
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 * 
 *       http://www.apache.org/licenses/LICENSE-2.0
 * 
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *  under the License.
 */
package nu.zoom.ldap.eon.connection.dns;

import java.awt.EventQueue;
import java.net.InetAddress;
import java.util.Hashtable;
import java.util.List;
import java.util.Map.Entry;
import java.util.Set;
import nu.zoom.ldap.eon.operation.Operation;
import nu.zoom.ldap.eon.operation.OperationManager;
import nu.zoom.swing.desktop.Workbench;
import nu.zoom.swing.table.StringTableModel;
import nu.zoom.util.dns.Resolver;
import nu.zoom.util.dns.SRVRecord;
import org.ops4j.gaderian.Messages;

/**
 *
 * @author joma7188
 */
public class FindServersPanel extends javax.swing.JPanel {

    private final Messages messages;
    private final OperationManager operationManager;
    private final Workbench workbench;
    private final Resolver resolver;
    private final StringTableModel resultTableModel;

    /** Creates new form FindServersPanel */
    public FindServersPanel(
            final Messages messages,
            final OperationManager operationManager,
            final Workbench workbench,
            final Resolver resolver) {
        this.messages = messages;
        this.operationManager = operationManager;
        this.workbench = workbench;
        this.resolver = resolver;

        final String[] columnNames = new String[5];
        columnNames[0] = messages.getMessage("connection.dns.result.table.host");
        columnNames[1] = messages.getMessage("connection.dns.result.table.address");
        columnNames[2] = messages.getMessage("connection.dns.result.table.port");
        columnNames[3] = messages.getMessage("connection.dns.result.table.prio");
        columnNames[4] = messages.getMessage("connection.dns.result.table.weight");
        resultTableModel = new StringTableModel(columnNames);

        initComponents();

        operationManager.runOperation(new CanonicalHostNameOperation());
    }

    class CanonicalHostNameOperation implements Operation {

        @Override
        public void execute() throws Exception {
            EventQueue.invokeLater(new Runnable() {

                @Override
                public void run() {
                    lookupProgressBar.setIndeterminate(true);
                    workbench.startWorkIndicator();
                    workbench.setStatusbarMessage(messages.getMessage("connection.dns.canonicalhostname"));
                }
            });
            final String canonicalHostName = InetAddress.getLocalHost().getCanonicalHostName();
            EventQueue.invokeLater(new Runnable() {

                @Override
                public void run() {
                    lookupProgressBar.setIndeterminate(false);
                    workbench.stopWorkIndicator();
                    workbench.setStatusbarMessage(canonicalHostName);
                    if ((domainNameField.getText() != null) && (domainNameField.getText().length() < 1)) {
                        domainNameField.setText(canonicalHostName);
                    }
                }
            });
        }
    }

    class FindADServersOperation implements Operation {

        private final String lookupName;

        public FindADServersOperation(final String lookupName) {
            this.lookupName = lookupName;
        }

        @Override
        public void execute() throws Exception {
            final Hashtable<SRVRecord, List<InetAddress>> result = new Hashtable<SRVRecord, List<InetAddress>>();
            try {
                List<SRVRecord> srvRecords = resolver.lookupSRV(lookupName);
                for (SRVRecord srvRecord : srvRecords) {
                    List<InetAddress> addresses = resolver.lookupA(srvRecord.getHostname());
                    result.put(srvRecord, addresses);
                }
            } finally {
                EventQueue.invokeLater(new Runnable() {

                    @Override
                    public void run() {
                        lookupProgressBar.setIndeterminate(false);
                        workbench.stopWorkIndicator();

                        if (result.size() < 1) {
                            workbench.setStatusbarMessage(messages.format("connection.dns.noresults", lookupName));
                        } else {
                            Set<Entry<SRVRecord, List<InetAddress>>> resultSet =
                                    result.entrySet();
                            for (Entry<SRVRecord, List<InetAddress>> entry : resultSet) {
                                SRVRecord srv = entry.getKey();
                                resultTableModel.addEntry(srv.getHostname());

                                List<InetAddress> addresses = entry.getValue();
                                String addressesString = "";
                                for (InetAddress address : addresses) {
                                    addressesString += " " + address.getHostAddress();
                                }
                                resultTableModel.addEntry(addressesString);

                                resultTableModel.addEntry("" + srv.getPort());
                                resultTableModel.addEntry("" + srv.getPriority());
                                resultTableModel.addEntry("" + srv.getWeight());
                            }
                        }
                        lookupButton.setEnabled(true);
                    }
                });
            }
        }
    }

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        domainNameField = new javax.swing.JTextField();
        domainNameFieldLabel = new javax.swing.JLabel();
        resultScrollPane = new javax.swing.JScrollPane();
        resultTable = new javax.swing.JTable();
        lookupProgressBar = new javax.swing.JProgressBar();
        lookupButton = new javax.swing.JButton();

        domainNameFieldLabel.setLabelFor(domainNameField);
        domainNameFieldLabel.setText(messages.getMessage("connection.dns.domainlabel"));

        resultTable.setModel(resultTableModel);
        resultScrollPane.setViewportView(resultTable);

        lookupButton.setText(messages.getMessage("connection.dns.domainbutton"));
        lookupButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                lookupButtonActionPerformed(evt);
            }
        });

        javax.swing.GroupLayout layout = new javax.swing.GroupLayout(this);
        this.setLayout(layout);
        layout.setHorizontalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, layout.createSequentialGroup()
                .addContainerGap()
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.TRAILING)
                    .addComponent(resultScrollPane, javax.swing.GroupLayout.Alignment.LEADING, javax.swing.GroupLayout.DEFAULT_SIZE, 394, Short.MAX_VALUE)
                    .addGroup(javax.swing.GroupLayout.Alignment.LEADING, layout.createSequentialGroup()
                        .addComponent(domainNameFieldLabel)
                        .addGap(18, 18, 18)
                        .addComponent(domainNameField, javax.swing.GroupLayout.DEFAULT_SIZE, 174, Short.MAX_VALUE)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
                        .addComponent(lookupButton))
                    .addComponent(lookupProgressBar, javax.swing.GroupLayout.DEFAULT_SIZE, 394, Short.MAX_VALUE))
                .addContainerGap())
        );
        layout.setVerticalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(layout.createSequentialGroup()
                .addContainerGap()
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(domainNameFieldLabel)
                    .addComponent(domainNameField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(lookupButton))
                .addGap(18, 18, 18)
                .addComponent(resultScrollPane, javax.swing.GroupLayout.DEFAULT_SIZE, 199, Short.MAX_VALUE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
                .addComponent(lookupProgressBar, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addContainerGap())
        );
    }// </editor-fold>//GEN-END:initComponents

    private void lookupButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_lookupButtonActionPerformed
        lookupProgressBar.setIndeterminate(true);
        workbench.startWorkIndicator();
        String lookup = "_ldap._tcp." + domainNameField.getText();
        String message = messages.format("connection.dns.lookup", lookup);
        workbench.setStatusbarMessage(message);
        lookupButton.setEnabled(false);
        resultTableModel.clear();
        operationManager.runOperation(new FindADServersOperation(lookup));
    }//GEN-LAST:event_lookupButtonActionPerformed
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JTextField domainNameField;
    private javax.swing.JLabel domainNameFieldLabel;
    private javax.swing.JButton lookupButton;
    private javax.swing.JProgressBar lookupProgressBar;
    private javax.swing.JScrollPane resultScrollPane;
    private javax.swing.JTable resultTable;
    // End of variables declaration//GEN-END:variables
}
