/*
 * Copyright (C) 2005 Johan Maasing johan at zoom.nu Licensed under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0 Unless required by applicable law
 * or agreed to in writing, software distributed under the License is
 * distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package nu.zoom.ldap.eon.connection;

import java.awt.EventQueue;

import javax.swing.Icon;
import javax.swing.JMenu;

import nu.zoom.ldap.eon.desktop.ComponentFactory;
import nu.zoom.ldap.eon.operation.OperationManager;
import nu.zoom.swing.desktop.Workbench;
import nu.zoom.swing.desktop.common.BackendException;

import org.ops4j.gaderian.Messages;

/**
 * Builds and rebuilds a connection menu.
 * 
 * @author $Author: johan $
 * @version $Revision: 1.3 $
 */
public class ConnectionMenuChangeHandler implements ConnectionFactoryListener {

    private JMenu connectionFactoryMenu = null;
    private Workbench workbench;
    private Messages messages;
    private ConnectionFactory connectionFactory;
    private ComponentFactory componentFactory;
    private OperationManager operationManager;

    public ConnectionMenuChangeHandler(Workbench workbench, Messages messages,
            ComponentFactory componentFactory,
            OperationManager operationManager,
            ConnectionFactory connectionFactory) {
        super();
        this.workbench = workbench;
        this.messages = messages;
        this.connectionFactory = connectionFactory;
        this.componentFactory = componentFactory;
        this.operationManager = operationManager;
        connectionFactory.addConnectionFactoryListener(this);
        rebuildMenu();
    }

    @Override
    public synchronized void createdConnection(Connection connection) {
        rebuildMenu();
    }

    @Override
    public synchronized void removedConnection(ConnectionGUID connectionGUID) {
        rebuildMenu();
    }

    @Override
    public synchronized void connectionDataChanged(Connection connection) {
        rebuildMenu();
    }

    synchronized JMenu getMenu() {
        return connectionFactoryMenu;
    }

    private synchronized void rebuildMenu() {
        if (EventQueue.isDispatchThread()) {
            rebuildMenuOnEventQueue();
        } else {
            EventQueue.invokeLater(new Runnable() {

                @Override
                public void run() {
                    rebuildMenuOnEventQueue();
                }
            });
        }
    }

    private synchronized void rebuildMenuOnEventQueue() {
        if (connectionFactoryMenu == null) {
            connectionFactoryMenu = new JMenu(connectionFactory.getName());
            connectionFactoryMenu.setToolTipText(connectionFactory.getDescription());
            Icon connectionIcon = connectionFactory.getIcon();
            if (connectionIcon != null) {
                connectionFactoryMenu.setIcon(connectionIcon);
            }
        } else {
            connectionFactoryMenu.removeAll();
        }

        try {
            Connection[] connections = connectionFactory.listConnections();
            if (connections != null) {
                for (int j = 0; j < connections.length; j++) {
                    JMenu connectionMenu = new JMenu(connections[j].getDescription());
                    connectionMenu.add(new ConnectAction(workbench, messages,
                            connections[j], componentFactory,
                            operationManager));
                    connectionMenu.add(new EditConnectionAction(workbench,
                            messages, connectionFactory, connections[j]));
                    connectionMenu.add(new DeleteConnectionAction(messages,
                            workbench, connectionFactory, connections[j]));
                    connectionFactoryMenu.add(connectionMenu);
                }
            }
        } catch (BackendException exc) {
            workbench.getErrorReporter().reportError(
                    messages.getMessage("connection.error.listconnection"),
                    exc);
        }

        connectionFactoryMenu.addSeparator();
        connectionFactoryMenu.add(new CreateConnectionAction(workbench,
                messages, connectionFactory, connectionFactory.getIcon()));

    }
}
