/*
 * Copyright (C) 2005 Johan Maasing johan at zoom.nu Licensed under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0 Unless required by applicable law
 * or agreed to in writing, software distributed under the License is
 * distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package nu.zoom.ldap.eon.connection;

import java.util.ArrayList;

import javax.swing.ImageIcon;

import nu.zoom.swing.desktop.common.BackendException;

/**
 * Abstract utility class for implementing Connection factories. This class
 * handles registration and informing listeners of factory events. Subclasses
 * are responsible for persisting the changes to connections.
 * 
 * @author $Author: johan $
 * @version $Revision: 1.4 $
 */
public abstract class AbstractConnectionFactory implements ConnectionFactory {

    private ArrayList<ConnectionFactoryListener> listeners = new ArrayList<ConnectionFactoryListener>();

    public AbstractConnectionFactory() {
        super();
    }

    @Override
    public synchronized void addConnectionFactoryListener(
            ConnectionFactoryListener listener) {
        if (listener != null) {
            if (!listeners.contains(listener)) {
                listeners.add(listener);
            }
        }
    }

    @Override
    public synchronized void removeConnectionFactoryListener(
            ConnectionFactoryListener listener) {
        listeners.remove(listener);
    }

    /**
     * Removes the connection (by delegating to a subclass implementation) and
     * informs the listeners that the connection has been removed.
     *
     * @see #delegateRemoveConnection(Connection)
     */
    @Override
    public synchronized void removeConnection(Connection connection) throws BackendException {
        if (connection != null) {
            ConnectionGUID id = connection.getGUID();
            delegateRemoveConnection(connection);
            fireConnectionRemoved(id);
        }
    }

    /**
     * Add a new connection to the list of connections. Informs listerners of
     * the change then delegates (to the subclass) the operation of adding the
     * connection.
     *
     * @see #delegateAddConnection(Connection)
     * @param connection
     *            The new connection to add.
     * @throws BackendException
     */
    public synchronized void addConnection(Connection connection)
            throws BackendException {
        delegateAddConnection(connection);
        fireConnectionAdded(connection);
    }

    /**
     * A connection has had its data changed. Informs listeners of the change
     * and delegates the change operation to the subclass.
     *
     * @param connection
     *            The connection that changed.
     * @throws BackendException
     */
    public synchronized void connectionChanged(Connection connection)
            throws BackendException {
        delegateConnectionChanged(connection);
        fireConnectionChanged(connection);
    }

    /**
     * Subclasses should add the connection to the list of connections. This
     * (parent) class will inform listeners that it has been added.
     *
     * @param connection
     */
    protected abstract void delegateAddConnection(Connection connection)
            throws BackendException;

    /**
     * Subclasses should remove the connection from the connection list. This
     * (parent) class will inform listeners that the connection has been
     * removed.
     *
     * @see #fireConnectionRemoved(ConnectionGUID)
     * @param connection
     *            The connection to remove. Never null.
     */
    protected abstract void delegateRemoveConnection(Connection connection)
            throws BackendException;

    /**
     * Subclasses should rememeber the new connection data.
     *
     * @param connection
     *            The connection that changed.
     */
    protected abstract void delegateConnectionChanged(Connection connection)
            throws BackendException;

    /*
     * (non-Javadoc)
     *
     * @see nu.zoom.ldap.eon.connection.ConnectionFactory#getDescription()
     */
    public abstract String getDescription();

    /*
     * (non-Javadoc)
     *
     * @see nu.zoom.ldap.eon.connection.ConnectionFactory#getName()
     */
    public abstract String getName();

    /*
     * (non-Javadoc)
     *
     * @see nu.zoom.ldap.eon.connection.ConnectionFactory#getIcon()
     */
    public abstract ImageIcon getIcon();

    /**
     * Subclasses should provide an editor for the type of connection it
     * handles. If a connection is added or changed the editor should call the
     * fireXXX methods on this class to inform connection factory listeners that
     * the list of connections has changed.
     *
     * @see #fireConnectionAdded(Connection)
     * @see #fireConnectionChanged(Connection)
     */
    public abstract ConnectionEditor getEditor();

    /**
     * Informs listeners that a connection has been added.
     *
     * @param connection
     *            The new connection.
     */
    protected synchronized void fireConnectionAdded(Connection connection) {
        for (ConnectionFactoryListener listener : listeners) {
            listener.createdConnection(connection);
        }
    }

    /**
     * Informs listeners that a connection has changed.
     *
     * @param connection
     *            The connection that changed.
     */
    protected synchronized void fireConnectionChanged(Connection connection) {
        for (ConnectionFactoryListener listener : listeners) {
            listener.connectionDataChanged(connection);
        }
    }

    /**
     * Informs listeners that a connection has been removed.
     *
     * @param connectionID
     *            The ID of the connection that was removed.
     */
    protected synchronized void fireConnectionRemoved(
            ConnectionGUID connectionID) {
        for (ConnectionFactoryListener listener : listeners) {
            listener.removedConnection(connectionID);
        }
    }
}
