/*
 * Copyright (C) 2005 Johan Maasing johan at zoom.nu Licensed under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0 Unless required by applicable law
 * or agreed to in writing, software distributed under the License is
 * distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package nu.zoom.ldap.eon.clipboard.impl;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedList;

import nu.zoom.ldap.eon.clipboard.DirectoryClipboard;
import nu.zoom.ldap.eon.clipboard.DirectoryClipboardItem;
import nu.zoom.ldap.eon.clipboard.DirectoryClipboardListener;
import nu.zoom.ldap.eon.directory.event.DirectoryEventListenerRegistry;
import nu.zoom.ldap.eon.directory.tree.DirectoryTreeObject;
import nu.zoom.swing.desktop.Workbench;
import nu.zoom.swing.desktop.common.BackendException;

import org.ops4j.gaderian.Messages;

/**
 * @version $Revision: 1.9 $
 * @author $Author: johan $
 */
public class DirectoryClipboardImpl implements DirectoryClipboard {

	private ArrayList<DirectoryClipboardListener> listeners = new ArrayList<DirectoryClipboardListener>();

	private LinkedList<DirectoryClipboardItem> clipboard = new LinkedList<DirectoryClipboardItem>();

	private Messages messages;

	private Workbench workbench;

	private DirectoryEventListenerRegistry eventRegistry;

	public DirectoryClipboardImpl(Messages messages, Workbench workbench,
			DirectoryEventListenerRegistry eventRegistry) {
		super();
		this.messages = messages;
		this.workbench = workbench;
		this.eventRegistry = eventRegistry;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see nu.zoom.ldap.eon.clipboard.DirectoryClipboard#addCopy(java.lang.String,
	 *      javax.naming.ldap.InitialLdapContext, javax.naming.Name)
	 */
	public synchronized void addCopy(DirectoryTreeObject sourceObject) {
		clipboard.add(new DirectoryObjectCopyClipboardItem(sourceObject,
				messages, workbench, eventRegistry));
		fireClipboardChanged();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see nu.zoom.ldap.eon.clipboard.DirectoryClipboard#addCut(java.lang.String,
	 *      javax.naming.ldap.InitialLdapContext, javax.naming.Name)
	 */
	public synchronized void addCut(DirectoryTreeObject sourceObject) {
		clipboard.add(new DirectoryObjectCutClipboardItem(sourceObject,
				messages, workbench, eventRegistry));
		fireClipboardChanged();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see nu.zoom.ldap.eon.clipboard.DirectoryClipboard#removeClipboardItem(java.lang.String)
	 */
	public synchronized void removeClipboardItem(String id) {
		if (id == null) {
			throw new IllegalArgumentException("ID can not be null");
		}
		Iterator clipboardIterator = clipboard.iterator();
		boolean foundIt = false;
		while (!foundIt && clipboardIterator.hasNext()) {
			DirectoryClipboardItem item = (DirectoryClipboardItem) clipboardIterator
					.next();
			if ((item != null) && (id.equals(item.getID()))) {
				foundIt = true;
				clipboard.remove(item);
				fireClipboardChanged();
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see nu.zoom.ldap.eon.clipboard.DirectoryClipboard#triggerPasteAction(java.lang.String,
	 *      javax.naming.ldap.InitialLdapContext, javax.naming.Name)
	 */
	public synchronized void triggerPasteAction(DirectoryTreeObject targetObject)
			throws BackendException {
		if (clipboard.size() > 0) {
			DirectoryClipboardItem item = (DirectoryClipboardItem) clipboard
					.removeFirst();
			fireClipboardChanged();
			if (item instanceof DirectoryObjectCopyClipboardItem) {
				((DirectoryObjectCopyClipboardItem) item).execute(targetObject);
			} else if (item instanceof DirectoryObjectCutClipboardItem) {
				((DirectoryObjectCutClipboardItem) item).execute(targetObject);
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see nu.zoom.ldap.eon.clipboard.DirectoryClipboard#addDirectoyClipboardListener(nu.zoom.ldap.eon.clipboard.DirectoryClipboardListener)
	 */
	public synchronized void addDirectoyClipboardListener(
			DirectoryClipboardListener listener) {
		if (listener != null) {
			listeners.add(listener);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see nu.zoom.ldap.eon.clipboard.DirectoryClipboard#removeDirectoyClipboardListener(nu.zoom.ldap.eon.clipboard.DirectoryClipboardListener)
	 */
	public synchronized void removeDirectoyClipboardListener(
			DirectoryClipboardListener listener) {
		listeners.remove(listener);
	}

	private synchronized void fireClipboardChanged() {
		for (DirectoryClipboardListener listener : listeners) {
			listener.clipboardContentChanged();
		}
	}
}
