/*
 *  Copyright 2009 joma7188.
 * 
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 * 
 *       http://www.apache.org/licenses/LICENSE-2.0
 * 
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *  under the License.
 */
package nu.zoom.util.dns;

/**
 *
 * @author joma7188
 */
public class SRVRecord implements Comparable<SRVRecord> {

    private final String hostname;
    private final int port;
    private final int priority;
    private final int weight;

    /**
     * Create a new record holding information about a SRV reply.
     * @param hostname The hostname in the DNS entry. May not be null.
     * @param port THe port number in the DNS entry.
     * @param priority The priority in the DNS entry.
     * @param weight THe weight in the DNS entry.
     */
    public SRVRecord(final String hostname, final int port, final int priority, final int weight) {
        if (hostname == null) {
            throw new IllegalArgumentException("Hostname may not be null");
        }
        this.hostname = hostname;
        this.priority = priority;
        this.weight = weight;
        this.port = port;
    }

    public String getHostname() {
        return hostname;
    }

    public int getPriority() {
        return priority;
    }

    public int getWeight() {
        return weight;
    }

    public int getPort() {
        return port;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final SRVRecord other = (SRVRecord) obj;
        if ((this.hostname == null) ? (other.hostname != null) : !this.hostname.equals(other.hostname)) {
            return false;
        }
        if (this.port != other.port) {
            return false;
        }
        if (this.priority != other.priority) {
            return false;
        }
        if (this.weight != other.weight) {
            return false;
        }
        return true;
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 59 * hash + (this.hostname != null ? this.hostname.hashCode() : 0);
        hash = 59 * hash + this.port;
        hash = 59 * hash + this.priority;
        hash = 59 * hash + this.weight;
        return hash;
    }

    @Override
    public String toString() {
        return hostname + ":" + port + " w:" + weight + " p:" + priority;
    }

    @Override
    public int compareTo(SRVRecord other) {
        // Delegate to String comparison (Cheap shot but works)
        String s = hostname + " " + port + " " + priority + " " + weight;
        String o = other.hostname + " " + other.port + " " + other.priority + " " + other.weight;
        return s.compareTo(o);
    }
}
