/*
 *  Copyright 2010 johan.
 * 
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 * 
 *       http://www.apache.org/licenses/LICENSE-2.0
 * 
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *  under the License.
 */
package nu.zoom.swing.action;

import java.awt.event.KeyEvent;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.Icon;
import javax.swing.KeyStroke;

/**
 * Methods for setting values on an action. Used to set tooltip, key bindings,
 * icons et c. for an action.
 * 
 * @author "Johan Maasing" &lt;johan@zoom.nu&gt;
 */
public abstract class AbstractTypedAction extends AbstractAction {
	private static final long serialVersionUID = -3956387600680656685L;

	/**
	 * Construct a new action. Remember to set the action name.
	 * 
	 * @see #setName(java.lang.String)
	 */
	public AbstractTypedAction() {
		super("<<<name not set>>>");
	}

	/**
	 * Set the name of the action.
	 * 
	 * @see Action#NAME
	 * @param name
	 *            The name to use.
	 */
	public void setName(String name) {
		putValue(Action.NAME, name);
	}

	/**
	 * Set the tooltip text.
	 * 
	 * @see Action#SHORT_DESCRIPTION
	 * @param tooltip
	 *            The tool tip text to use.
	 */
	public void setToolTip(String tooltip) {
		putValue(Action.SHORT_DESCRIPTION, tooltip);
	}

	/**
	 * Set the accelerator key. An accelerator is a key combination that causes
	 * a menu item to be chosen, whether or not it's visible. See <a href=
	 * "http://java.sun.com/docs/books/tutorial/uiswing/components/menu.html">
	 * How to use menues </a>
	 * 
	 * @see KeyEvent
	 * @see Action#ACCELERATOR_KEY
	 * @see KeyStroke#getKeyStroke(int, int)
	 * @param keyEvent
	 *            The key used to trigger the action. For example KeyEvent.VK_L
	 * @param keyEventMask
	 *            The key mask (for example KeyEvent.CTRL_DOWN_MASK)).
	 */
	public void setAcceleratorKey(int keyEvent, int keyEventMask) {
		putValue(Action.ACCELERATOR_KEY, KeyStroke.getKeyStroke(keyEvent, keyEventMask));
	}

	/**
	 * Set the Mnemonic key. A mnemonic is a key that makes an already visible
	 * menu item be chosen. See <a href=
	 * "http://java.sun.com/docs/books/tutorial/uiswing/components/menu.html">
	 * How to use menues </a>
	 * 
	 * @see KeyEvent
	 * @see Action#MNEMONIC_KEY
	 * @param keyEvent
	 *            The key used to trigger the action.
	 */
	public void setMnemonicKey(int keyEvent) {
		putValue(Action.MNEMONIC_KEY, new Integer(keyEvent));
	}

	/**
	 * Set the icon for the action.
	 * 
	 * @see Action#SMALL_ICON
	 * @param icon
	 *            The icon to use.
	 */
	public void setIcon(Icon icon) {
		if (icon != null) {
			putValue(Action.SMALL_ICON, icon);
		}
	}
}
