/*
 * Copyright (C) 2004 Johan Maasing johan at zoom.nu Licensed under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0 Unless required by applicable law
 * or agreed to in writing, software distributed under the License is
 * distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package nu.zoom.swing;

import java.util.ArrayList;

import javax.swing.filechooser.FileFilter;

/**
 * A file name filter for use with JFileChooser, filters on file extensions. Use
 * it like this, instantiate a new nu.zoom.ExtensionFilter, tell the filter
 * which extensions you want displayed in the JFileChooser. Set the JFileChooser
 * to use the filter. That's it. Optionally you can set a short description for
 * the files. <br />
 * <code>import nu.zoom.swing.ExtensionFilter ;</code><br />
 * <br />
 * <code>ExtensionFilter filter = new ExtensionFilter() ;</code><br />
 * <code>filter.addExtension(".gif") ;</code><br />
 * <code>filter.addExtension(".jpg") ;</code><br />
 * <code>filter.setDescription("Image files");</code><br />
 * <code>JFileChooser chooser = new JFileChooser() ;</code><br />
 * <code>chooser.setFileFilter(filter) ;</code><br />
 * <code>chooser.showOpenDialog(parent) ;</code><br />
 * 
 * Basically this does the same job as the ExtensionFileFilter SUN talks about
 * in the JDK1.2 documentation. Strange enough the SUN documentation says that
 * class will be included when Swing goes to 1.0.3 but I can't find it in swing
 * 1.1.
 * 
 * @author $Author: johan $
 * @version $Revision: 1.2 $
 */
public class ExtensionFilter extends FileFilter implements
		java.io.FilenameFilter
{
	/**
	 * Default constructor. Intializes a filter that does not display any files.
	 */
	private ArrayList<String> extensions;
	private String description;

	public ExtensionFilter() {
		super();
		extensions = new ArrayList<String>();
	}

	/**
	 * Adds a file extension that we want the JFileChooser to display. Notice
	 * that the filter will match any filename that ends with the extension
	 * string. This means the you probably want to set the extension string with
	 * a leading dot - ".gif" instead of "gif". This also means that an
	 * extension of ".tar.gz" can be used in the filter to disregard "johnny.gz"
	 * but to show the file named "johnny.tar.gz". The comparison ignores case.
	 */
	public void addExtension(String extension)
	{
		if ((extension != null) && (extension.length() > 0)) {
			extensions.add(extension);
		}
	}

	/**
	 * Sets the description string used by the JFileChooser to inform the user
	 * of what kind of files are displayed.
	 */
	public void setDescription(String description)
	{
		this.description = new String(description);
	}

	/**
	 * Internally used by JFileChooser to filter files.
	 */
	public boolean accept(java.io.File f)
	{
		// If it is a directory it should be visible.
		if (f.isDirectory())
			return true;
		// Convert the incoming file name to uppercase to easy the string
		// comparison.
		String fileExtension = f.getName();
		fileExtension = fileExtension.toUpperCase();
		// Compare the incoming file name with the stored extensions
		for (String extension : extensions) {
			extension = extension.toUpperCase();
			// Incoming file name matches one of the extensions
			if (fileExtension.endsWith(extension))
				return true;
		}
		return false;
	}

	/**
	 * Internally used by JFileChooser, returns the description string.
	 */
	public String getDescription()
	{
		return description;
	}

	/**
	 * FilenameFilter, ignores the directory and only looks for the extension.
	 */
	public boolean accept(java.io.File directory, String name)
	{
		// Convert the incoming file name to uppercase to ease the string
		// comparison.
		String fileExtension = name.toUpperCase();
		for (String extension : extensions) {
			extension = extension.toUpperCase();
			// Incoming file name matches one of the extensions
			if (fileExtension.endsWith(extension))
				return true;
		}
		return false;
	}
}