/*
 * Copyright (C) 2005 Johan Maasing johan at zoom.nu Licensed under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0 Unless required by applicable law
 * or agreed to in writing, software distributed under the License is
 * distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package nu.zoom.swing.desktop.plugin.console;

import java.awt.Color;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.ImageIcon;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.KeyStroke;

import nu.zoom.swing.desktop.Workbench;
import nu.zoom.swing.desktop.WorkbenchFrame;
import nu.zoom.swing.desktop.WorkbenchFrameListener;
import nu.zoom.swing.desktop.WorkbenchListener;
import nu.zoom.swing.desktop.common.BackendException;
import nu.zoom.swing.desktop.preferences.Preferences;
import nu.zoom.swing.text.ConsoleBean;
import nu.zoom.swing.text.ConsoleBeanBeanInfo;

import org.ops4j.gaderian.Messages;

/**
 * @version $Revision: 1.2 $
 * @author $Author: johan $
 */
public class ConsoleImpl implements WorkbenchListener, Console {

	private static final String IS_CONSOLE_WINDOW_VISIBLE = "nu.zoom.desktop.plugin.console";

	private Messages messages;

	private Workbench workbench;

	private WorkbenchFrame consoleFrame = null;

	private Preferences preferences;

	private ConsoleBean consoleBean = new ConsoleBean();

	private JCheckBoxMenuItem consoleMenuItem;

	private boolean closing = false;

	/**
	 * 
	 */
	public ConsoleImpl(Workbench workbench, Preferences preferences) {
		super();
		this.workbench = workbench;
		this.preferences = preferences;
	}

	/**
	 * Sets the Hivemind messages
	 * 
	 * @param messages
	 *            The messages to set.
	 */
	public void setMessages(Messages messages) {
		this.messages = messages;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see nu.zoom.swing.desktop.plugin.Console#getConsole()
	 */
	public ConsoleBean getConsole() {
		return consoleBean;
	}

	/**
	 * 
	 */
	private void buildGUI() {
		consoleBean.setInfoColor(new Color(0, 128, 0));
		ConsoleBeanAction consoleBeanAction = new ConsoleBeanAction();
		consoleMenuItem = new JCheckBoxMenuItem(consoleBeanAction);
		workbench.getMenuBar().addToApplicationMenu(consoleMenuItem);
		try {
			// open previously open windows
			if (isConsoleWindowVisible()) {
				consoleBeanAction.showConsoleFrame();
			} else {
				consoleMenuItem.setSelected(false);
			}
		} catch (BackendException e) {
			String message = messages.format("error.preferences", e.toString());
			consoleBean.error(message);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see nu.zoom.swing.desktop.gui.WorkbenchListener#close()
	 */
	public void close() {
		closing = true;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see nu.zoom.swing.desktop.PlugIn#initialize()
	 */
	public void initialize() throws Exception {
		workbench.addWorkBenchListener(this);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see nu.zoom.swing.desktop.gui.WorkbenchListener#start()
	 */
	public void start() {
		buildGUI();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see nu.zoom.ca.gui.ApplicationPreferences#isConsoleWindowVisible()
	 */
	private boolean isConsoleWindowVisible() throws BackendException {
		try {
			String state = preferences.getString(IS_CONSOLE_WINDOW_VISIBLE);
			return (state == null) ? false : Boolean.valueOf(state)
					.booleanValue();
		} catch (Exception e) {
			throw new BackendException(e);
		}
	}

	private void setPreferenceWindowVisible(boolean isVisible)
			throws BackendException {
		try {
			preferences.setString(IS_CONSOLE_WINDOW_VISIBLE, Boolean
					.toString(isVisible));
		} catch (Exception e) {
			throw new BackendException(e);
		}
	}

	@SuppressWarnings("serial")
	private class ConsoleBeanAction extends AbstractAction implements
			WorkbenchFrameListener {

		private ImageIcon consoleIcon;

		ConsoleBeanAction() {
			super(messages.getMessage("application.console"));
			consoleIcon = new ImageIcon(new ConsoleBeanBeanInfo()
					.getIcon(ConsoleBeanBeanInfo.ICON_COLOR_16x16));
			putValue(Action.SHORT_DESCRIPTION, messages
					.getMessage("application.console.tooltip"));
			putValue(Action.SMALL_ICON, consoleIcon);
			putValue(Action.ACCELERATOR_KEY, KeyStroke.getKeyStroke(
					KeyEvent.VK_C, KeyEvent.CTRL_DOWN_MASK));
			putValue(Action.MNEMONIC_KEY, new Integer(KeyEvent.VK_C));
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see nu.zoom.swing.desktop.gui.WorkbenchFrameListener#frameClosed(nu.zoom.swing.desktop.gui.WorkbenchFrame)
		 */
		public void frameClosed(WorkbenchFrame frame) {
			// If the frame got closed because the application is shutting down
			// We do not want to remember that the console was closed.
			// If it is not closing the frame got disposed of by the user.
			if (!closing) {
				consoleMenuItem.setSelected(false);
				try {
					setPreferenceWindowVisible(false);
				} catch (BackendException exc) {
					String message = messages.format("error.preferences", exc
							.toString());
					consoleBean.error(message);
				}
			}
			consoleFrame = null;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see nu.zoom.swing.desktop.WorkbenchFrameListener#frameWillDispose(nu.zoom.swing.desktop.WorkbenchFrame)
		 */
		public void frameWillDispose(WorkbenchFrame frame) {
		}

		private void showConsoleFrame() {
			consoleFrame = workbench.createWorkbenchFrame(this.getClass()
					.getName(), consoleBean, true, true);
			consoleFrame.addFrameListener(this);
			consoleFrame.setFrameIcon(consoleIcon);
			consoleFrame.setTitle(messages.getMessage("application.console"));
			consoleMenuItem.setSelected(true);
			consoleFrame.setVisible(true);
			try {
				setPreferenceWindowVisible(true);
			} catch (BackendException exc) {
				String message = messages.format("error.preferences", exc
						.toString());
				consoleBean.error(message);
			}
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
		 */
		public void actionPerformed(ActionEvent e) {
			if (consoleFrame == null) {
				showConsoleFrame();
			} else {
				consoleFrame.dispose();
			}
		}
	}
}
