/*
 * Copyright (C) 2005 Johan Maasing johan at zoom.nu Licensed under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0 Unless required by applicable law
 * or agreed to in writing, software distributed under the License is
 * distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package nu.zoom.swing.desktop.plugin.browser;

import java.awt.BorderLayout;
import java.io.IOException;
import java.net.URL;

import javax.swing.JEditorPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.event.HyperlinkEvent;
import javax.swing.event.HyperlinkListener;
import javax.swing.text.html.HTMLDocument;
import javax.swing.text.html.HTMLFrameHyperlinkEvent;

import nu.zoom.swing.desktop.Workbench;
import nu.zoom.swing.desktop.WorkbenchFrame;
import nu.zoom.swing.text.StatusBarBean;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ops4j.gaderian.Messages;

/**
 * @version $Revision: 1.2 $
 * @author $Author: johan $
 */
public class BrowserImpl implements Browser {
	private Log log = LogFactory.getLog(getClass()) ;

	private JTextField navField;

	private WorkbenchFrame frame = null;

	private Messages messages;

	private Workbench workbench;

	private JEditorPane pane = null;

	private StatusBarBean statusBar;

	public BrowserImpl(Workbench workbench, Messages messages) {
		super();
		this.workbench = workbench;
		this.messages = messages;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see nu.zoom.swing.desktop.plugin.browser.Browser#open(java.net.URL)
	 */
	public Browser open(URL url) {
		log.trace("Creating a new browser instance") ;
		Browser browser = new BrowserImpl(workbench, messages);
		browser.show(url);
		return browser;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see nu.zoom.swing.desktop.plugin.browser.Browser#show(java.net.URL)
	 */
	public void show(URL url) {
		log.trace("Trying to display: " + url) ;
		if (frame == null) {
			createFrame();
			frame.setVisible(true);
		}
		frame.moveToFront();
		statusBar.setMessage(messages.format("browser.statusbar.opening", url
				.toString()));
		navField.setText(url.toString());
		try {
			pane.setPage(url);
			statusBar.setMessage("Loaded");
		} catch (IOException e) {
			log.info("Unable to display: " + url, e);
			statusBar.setMessage(messages.getMessage("browser.error"));
			workbench.getErrorReporter().reportError(
					messages.getMessage("browser.error"), e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see nu.zoom.swing.desktop.PlugIn#initialize()
	 */
	public void initialize() throws Exception {
	}

	private void createFrame() {
		pane = new JEditorPane("text/html", "");
		pane.setEditable(false);
		navField = new JTextField(80);
		statusBar = new StatusBarBean();
		JPanel navigationpanel = new JPanel(new BorderLayout());
		navigationpanel.add(navField, BorderLayout.CENTER);

		JPanel mainPanel = new JPanel(new BorderLayout());
		mainPanel.add(navigationpanel, BorderLayout.NORTH);
		mainPanel.add(new JScrollPane(pane), BorderLayout.CENTER);
		mainPanel.add(statusBar, BorderLayout.SOUTH);
		frame = workbench.createWorkbenchFrame(this.getClass().getName(),
				mainPanel, true, true);
		frame.setTitle(messages.getMessage("browser.title"));
		pane.addHyperlinkListener(new Hyperactive());
	}

	class Hyperactive implements HyperlinkListener {

		public void hyperlinkUpdate(HyperlinkEvent e) {
			if (e.getEventType() == HyperlinkEvent.EventType.ACTIVATED) {
				JEditorPane pane = (JEditorPane) e.getSource();
				if (e instanceof HTMLFrameHyperlinkEvent) {
					HTMLFrameHyperlinkEvent evt = (HTMLFrameHyperlinkEvent) e;
					HTMLDocument doc = (HTMLDocument) pane.getDocument();
					doc.processHTMLFrameHyperlinkEvent(evt);
				} else {
					try {
						show(e.getURL());
					} catch (Exception exc) {
						workbench.getErrorReporter().reportError(exc);
					}
				}
			}
		}
	}

}
