package nu.zoom.swing.desktop.frame;

import java.util.ArrayList;
import java.util.UUID;

import javax.swing.ImageIcon;

import nu.zoom.swing.desktop.WorkbenchFrame;
import nu.zoom.swing.desktop.WorkbenchFrameListener;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public abstract class AbstractWorkbenchFrame implements WorkbenchFrame {
    private Log log = LogFactory.getLog(getClass());
    private final String guid;
    private final ArrayList<WorkbenchFrameListener> frameListeners = new ArrayList<WorkbenchFrameListener>();

    public AbstractWorkbenchFrame() {
        super();
        guid = UUID.randomUUID().toString();
    }

    @Override
    public synchronized void addFrameListener(WorkbenchFrameListener listener) {
        if ((listener != null) && (!frameListeners.contains(listener))) {
            frameListeners.add(listener);
        }
    }

    @Override
    public synchronized void removeFrameListener(WorkbenchFrameListener listener) {
        if (listener != null) {
            frameListeners.remove(listener);
        }
    }

    protected synchronized void fireFrameClosed() {
        if (log.isTraceEnabled()) {
            log.trace("Informing " + frameListeners.size()
                    + " listeners that frame is closed");
        }
        for (int f = 0; f < frameListeners.size(); f++) {
            ((WorkbenchFrameListener) frameListeners.get(f)).frameClosed(this);
        }
    }

    protected synchronized void fireFrameWillDispose() {
        if (log.isTraceEnabled()) {
            log.trace("Informing " + frameListeners.size()
                    + " listeners that frame will dispose");
        }
        for (int f = 0; f < frameListeners.size(); f++) {
            ((WorkbenchFrameListener) frameListeners.get(f)).frameWillDispose(this);
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see nu.zoom.swing.desktop.WorkbenchFrame#dispose()
     */
    @Override
    public void dispose() {
        fireFrameWillDispose();
        delegateDispose();
    }

    /**
     * Subclasses should dispose of their Swing/AWT frame and when the frame has
     * been disposed call {@link #fireFrameClosed()}
     *
     */
    protected abstract void delegateDispose();

    /*
     * (non-Javadoc)
     *
     * @see nu.zoom.swing.desktop.WorkbenchFrame#moveToFront()
     */
    @Override
    public abstract void moveToFront();

    /*
     * (non-Javadoc)
     *
     * @see nu.zoom.swing.desktop.WorkbenchFrame#setFrameIcon(javax.swing.Icon)
     */
    @Override
    public abstract void setFrameIcon(ImageIcon icon);

    /*
     * (non-Javadoc)
     *
     * @see nu.zoom.swing.desktop.WorkbenchFrame#getTitle()
     */
    @Override
    public abstract String getTitle();

    /*
     * (non-Javadoc)
     *
     * @see nu.zoom.swing.desktop.WorkbenchFrame#setTitle(java.lang.String)
     */
    @Override
    public abstract void setTitle(final String newTitle);

    /*
     * (non-Javadoc)
     *
     * @see nu.zoom.swing.desktop.WorkbenchFrame#setVisible(boolean)
     */
    @Override
    public abstract void setVisible(final boolean visible);

    /*
     * (non-Javadoc)
     *
     * @see nu.zoom.swing.desktop.WorkbenchFrame#isVisible()
     */
    @Override
    public abstract boolean isVisible();

    /* (non-Javadoc)
     * @see nu.zoom.swing.desktop.WorkbenchFrame#attach()
     */
    @Override
    public abstract void attach();

    /* (non-Javadoc)
     * @see nu.zoom.swing.desktop.WorkbenchFrame#detach()
     */
    @Override
    public abstract void detach();

    /* (non-Javadoc)
     * @see nu.zoom.swing.desktop.WorkbenchFrame#isDetached()
     */
    @Override
    public abstract boolean isDetached();

    @Override
    public boolean equals(Object obj) {
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final AbstractWorkbenchFrame other = (AbstractWorkbenchFrame) obj;
        if ((this.guid == null) ? (other.guid != null) : !this.guid.equals(other.guid)) {
            return false;
        }
        return true;
    }

    @Override
    public int hashCode() {
        int hash = 3;
        hash = 97 * hash + (this.guid != null ? this.guid.hashCode() : 0);
        return hash;
    }
}
