/*
 * Copyright (C) 2005 Johan Maasing johan at zoom.nu Licensed under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0 Unless required by applicable law
 * or agreed to in writing, software distributed under the License is
 * distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package nu.zoom.swing.desktop.common.action;

import java.awt.event.ActionEvent;
import java.net.URL;

import javax.swing.Action;
import javax.swing.ImageIcon;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ops4j.gaderian.Messages;

/**
 * Utility class to create Actions for a menu/toolbar. Has some set methods to
 * set tooltip, icon etc by looking up values in a HiveMind provided messages
 * class.
 * 
 * @see org.apache.hivemind.Messages
 * @author $Author: johan $
 * @version $Revision: 1.2 $
 */
public abstract class WorkbenchMessageAction extends WorkbenchAction
{
	private Log log = LogFactory.getLog(getClass());
	protected Messages messages;

	/**
	 * Construct a new action.
	 * 
	 * @param messages
	 *            The HiveMind messages bundle.
	 */
	public WorkbenchMessageAction(Messages messages) {
		super();
		this.messages = messages;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
	 */
	public abstract void actionPerformed(ActionEvent e);

	/**
	 * Sets the name of the action from the message bundle.
	 * 
	 * @see Action#NAME
	 * @param messageKey
	 *            The key used to lookup the value.
	 */
	public void setNameFromMessages(String messageKey)
	{
		setName(messages.getMessage(messageKey));
	}

	/**
	 * Set the tooltip text from the message bundle.
	 * 
	 * @see Action#SHORT_DESCRIPTION
	 * @param messageKey
	 *            The key used to lookup the value.
	 */
	public void setToolTipFromMessages(String messageKey)
	{
		setToolTip(messages.getMessage(messageKey));
	}

	/**
	 * Set the icon for the action. The icon will be loaded by the classloader
	 * using the resource name found in the message bundle under the given
	 * resourceNameKey.
	 * 
	 * @see Action#SMALL_ICON
	 * @param resourceNameKey
	 *            The key used to lookup the resource name.
	 */
	public void setIconFromMessages(String resourceNameKey)
	{
		String resourceName = messages.getMessage(resourceNameKey);
		log.trace("Trying to retrieve resource: " + resourceName) ;
		URL url = getClass().getResource(resourceName);
		ImageIcon icon = null;
		if (url != null) {
			icon = new ImageIcon(url);
			if (icon != null) {
				log.trace("Constructed an icon") ;
				putValue(Action.SMALL_ICON, icon);
			}
		}
	}

}
