/*
 * Copyright (C) 2005 Johan Maasing johan at zoom.nu Licensed under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0 Unless required by applicable law
 * or agreed to in writing, software distributed under the License is
 * distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package nu.zoom.swing.desktop.common.action;

import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.Icon;
import javax.swing.KeyStroke;

/**
 * Utility class to create Actions for a menu/toolbar. Has some set methods to
 * set tooltip, icon etc.
 * 
 * @see org.apache.hivemind.Messages
 * @author $Author: johan $
 * @version $Revision: 1.7 $
 */
public abstract class WorkbenchAction extends AbstractAction
{
	/**
	 * Construct a new action.
	 */
	public WorkbenchAction() {
		super("<<<name not set>>>");
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
	 */
	public abstract void actionPerformed(ActionEvent e);

	/**
	 * Set the name of the action.
	 * 
	 * @see Action#NAME
	 * @param name
	 *            The name to use.
	 */
	public void setName(String name)
	{
		putValue(Action.NAME, name);
	}

	/**
	 * Set the tooltip text.
	 * 
	 * @see Action#SHORT_DESCRIPTION
	 * @param tooltip
	 *            The tool tip text to use.
	 */
	public void setToolTip(String tooltip)
	{
		putValue(Action.SHORT_DESCRIPTION, tooltip);
	}

	/**
	 * Set the accelerator key. An accelerator is a key combination that causes
	 * a menu item to be chosen, whether or not it's visible. See <a
	 * href="http://java.sun.com/docs/books/tutorial/uiswing/components/menu.html">
	 * How to use menues </a>
	 * 
	 * @see KeyEvent
	 * @see Action#ACCELERATOR_KEY
	 * @param keyEvent
	 *            The key used to trigger the action. For example KeyEvent.VK_L
	 * @param keyEventMask
	 *            The key mask (for example KeyEvent.CTRL_DOWN_MASK)).
	 */
	public void setAcceleratorKey(int keyEvent, int keyEventMask)
	{
		putValue(Action.ACCELERATOR_KEY, KeyStroke.getKeyStroke(keyEvent,
				keyEventMask));
	}

	/**
	 * Set the Mnemonic key. A mnemonic is a key that makes an already visible
	 * menu item be chosen. See <a
	 * href="http://java.sun.com/docs/books/tutorial/uiswing/components/menu.html">
	 * How to use menues </a>
	 * 
	 * @see KeyEvent
	 * @see Action#MNEMONIC_KEY
	 * @param keyEvent
	 *            The key used to trigger the action.
	 */
	public void setMnemonicKey(int keyEvent)
	{
		putValue(Action.MNEMONIC_KEY, new Integer(keyEvent));
	}

	/**
	 * Set the icon for the action.
	 * 
	 * @see Action#SMALL_ICON
	 * @param icon
	 *            The icon to use.
	 */
	public void setIcon(Icon icon)
	{
		if (icon != null) {
			putValue(Action.SMALL_ICON, icon);
		}
	}
}
