/*
 * Copyright (C) 2005 Johan Maasing johan at zoom.nu Licensed under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0 Unless required by applicable law
 * or agreed to in writing, software distributed under the License is
 * distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package nu.zoom.swing.desktop;

import javax.swing.JMenu;
import javax.swing.JMenuItem;

/**
 * Wraps the Workbench menu bar. You add a menu to the application menu bar by
 * calling addMenu. All menus added must have an unique identification string.
 * This id string can be used to later retrieve the menu or remove the menu. The
 * menu bar has 2 predefined menues (that can not be removed). The application
 * menu that will always be on the far left and the Help menu that will always
 * be on the far right. Items can be added to the predefined menues by calling
 * the appropriate utility method, addToApplicationMenu or addToHelpMenu.
 * 
 * @version $Revision: 1.2 $
 * @author $Author: johan $
 */
public interface WorkbenchMenuBar
{

	public static final String HELP_MENU_KEY = "nu.zoom.swing.desktop.gui.workbench.menu.help";
	public static final String APPLICATION_MENU_KEY = "nu.zoom.swing.desktop.gui.workbench.menu.help";

	/**
	 * Get a menu on the menubar.
	 * 
	 * @param menuKey
	 *            The key which was used to add the menu to the menu bar.
	 * @return The menu or null if not found.
	 */
	public JMenu getMenu(String menuKey);

	/**
	 * Remove a menu from the menubar. Does nothing if the key is invalid or the
	 * menu not found.
	 * 
	 * @param menuKey
	 *            The key with which the menu was added to the menu bar.
	 */
	public void removeMenu(String menuKey);

	/**
	 * Add a menu to the menu bar, the menu will be added to the right of the
	 * current menus. The Help menu will however always be at the far right.
	 * 
	 * @param menuKey
	 *            The key to store the menu as (to identify it later). Can not
	 *            be null or zero length.
	 * @param menu
	 *            The menu to add, can not be null.
	 * @throws IllegalArgumentException
	 *             If the key or the menu is null.
	 */
	public void addMenu(String menuKey, JMenu menu)
			throws IllegalArgumentException;

	/**
	 * Add a menu item to the Application menu. This item will be placed first
	 * on the application menu.
	 * 
	 * @param item
	 *            The menu item to put in the application menu. Not null.
	 * @throws IllegalArgumentException
	 *             If the item is null.
	 */
	public void addToApplicationMenu(JMenuItem item)
			throws IllegalArgumentException;

	/**
	 * Add a menu item to the Application menu. This item will be appended to
	 * the help menu.
	 * 
	 * @param item
	 *            The menu item to put in the help menu. Not null.
	 * @throws IllegalArgumentException
	 *             If the item is null.
	 */
	public void addToHelpMenu(JMenuItem item) throws IllegalArgumentException;
}