/*
 * Copyright (C) 2004 Johan Maasing johan at zoom.nu Licensed under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0 Unless required by applicable law
 * or agreed to in writing, software distributed under the License is
 * distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package nu.zoom.swing.desktop;

import java.util.List;

import javax.swing.ImageIcon;

import nu.zoom.swing.desktop.worker.EventQueuePolicy;
import nu.zoom.swing.desktop.worker.Policy;

/**
 * The frame that is displayed on the workbench. This class is thread safe and
 * can be called from any thread (it does not have to be the AWT/Swing event
 * thread).
 * 
 * @see javax.swing.JInternalFrame
 * @author $Author: johan $
 * @version $Revision: 1.4 $
 */
public interface WorkbenchFrame {

	/**
	 * Add a listener for frame events.
	 * 
	 * @param listener
	 */
	public void addFrameListener(WorkbenchFrameListener listener);

	/**
	 * Remove a registered frame event listener
	 * 
	 * @param listener
	 */
	public void removeFrameListener(WorkbenchFrameListener listener);

	/**
	 * Close the frame and dispose of it.
	 */
	@EventQueuePolicy(Policy.EVENT_QUEUE)
	public void dispose();

	/**
	 * Set an icon for the frame. This is the ame as calling
	 * {@link #setFrameIcon(List)} with a list of one element.
	 * 
	 * @param icon
	 */
	@EventQueuePolicy(Policy.EVENT_QUEUE)
	public void setFrameIcon(ImageIcon icon);

	/**
	 * Set the icons for the frame.
	 * 
	 * @param icon
	 */
	@EventQueuePolicy(Policy.EVENT_QUEUE)
	public void setFrameIcon(List<ImageIcon> icons);

	/**
	 * Moves the frame to the front of the frame-stack
	 * 
	 * @see javax.swing.JInternalFrame#moveToFront()
	 */
	@EventQueuePolicy(Policy.EVENT_QUEUE)
	public void moveToFront();

	/**
	 * Sets the visible state of the frame.
	 * 
	 * @see javax.swing.JComponent#setVisible(boolean)
	 * @param visible
	 *            The visibel state the frame should have.
	 */
	@EventQueuePolicy(Policy.EVENT_QUEUE)
	public void setVisible(boolean visible);

	/**
	 * Indicates if the frame is explicitly hidden or not.
	 * 
	 * @see java.awt.Component#isVisible()
	 * @return The visibility state.
	 */
	@EventQueuePolicy(Policy.EVENT_QUEUE)
	public boolean isVisible();

	/**
	 * Set the title of the frame.
	 * 
	 * @param newTitle
	 */
	@EventQueuePolicy(Policy.EVENT_QUEUE)
	public void setTitle(String newTitle);

	/**
	 * Get the current frame title.
	 * 
	 * @return The current title. Returns an empty string if the title is not
	 *         set, never null.
	 */
	@EventQueuePolicy(Policy.EVENT_QUEUE)
	public String getTitle();

	/**
	 * Indicates if the frame is detached from the desktop as a top-level JFrame
	 * or not.
	 * 
	 * @return true if the frame lives outside the desktop, false if it is an
	 *         internal frame on the desktop.
	 */
	public boolean isDetached();

	/**
	 * Moves the frame from the desktop to a top-level JFrame. If the frame
	 * already is detached this method does nothing.
	 * 
	 */
	public void detach();

	/**
	 * Attach a detached frame to the desktop. That is make it en Internal frame
	 * again. If the frame already is attached this method does nothing.
	 * 
	 */
	public void attach();
}