package jibe.tools.testing.bdd.executions.web;

import com.google.common.base.Optional;
import jibe.tools.bdd.api.AbstractExecution;
import jibe.tools.bdd.api.ExecutionContext;
import org.openqa.selenium.WebDriver;
import org.openqa.selenium.phantomjs.PhantomJSDriver;
import org.openqa.selenium.remote.DesiredCapabilities;
import org.seleniumhq.selenium.fluent.FluentWebDriver;

import java.io.File;
import java.util.concurrent.TimeUnit;

import static org.openqa.selenium.phantomjs.PhantomJSDriverService.PHANTOMJS_EXECUTABLE_PATH_PROPERTY;

/**
 *
 */
abstract public class AbstractWebExecution<T> extends AbstractExecution {

    public static final String FLUENT_WEBDRIVER_KEY = AbstractWebExecution.class.getName() + ".fluentWebDriver";
    public static final String PHANTOMJS_BINARY_KEY = AbstractWebExecution.class.getName() + ".phantomJSBinary";
    public static final String PHANTOMJS_DRIVER_KEY = AbstractWebExecution.class.getName() + ".phantomJSDriver";

    @Override
    public void preExecute(ExecutionContext ctx) {
        if (haveFluentWebDriver(ctx)) {
            return;
        }

        Object phantomjsBinary = ctx.get(PHANTOMJS_BINARY_KEY);
        if ((phantomjsBinary instanceof String)) {
            phantomjsBinary = new File((String) phantomjsBinary);
        }

        ThreadLocal<PhantomJSDriver> phantomJSDriverThreadLocal = new ThreadLocal<>();
        phantomJSDriverThreadLocal.set(newPhantomJSDriver((File) phantomjsBinary));

        ThreadLocal<FluentWebDriver> fluentWebDriverThreadLocal = new ThreadLocal<>();
        fluentWebDriverThreadLocal.set(new FluentWebDriver(phantomJSDriverThreadLocal.get()));

        ctx.put(PHANTOMJS_DRIVER_KEY, phantomJSDriverThreadLocal);
        ctx.put(FLUENT_WEBDRIVER_KEY, fluentWebDriverThreadLocal);
    }

    @Override
    abstract public T execute(ExecutionContext ctx);

    protected FluentWebDriver getFluentWebDriver(ExecutionContext ctx) {
        return ((ThreadLocal<FluentWebDriver>) ctx.get(FLUENT_WEBDRIVER_KEY)).get();
    }

    protected WebDriver getWebDriver(ExecutionContext ctx) {
        return ((ThreadLocal<WebDriver>) ctx.get(PHANTOMJS_DRIVER_KEY)).get();
    }

    protected Optional<FluentWebDriver> findFluentWebDriver(ExecutionContext ctx) {
        return haveFluentWebDriver(ctx) ? Optional.of(getFluentWebDriver(ctx)) : Optional.<FluentWebDriver>absent();
    }

    protected Optional<WebDriver> findWebDriver(ExecutionContext ctx) {
        return haveFluentWebDriver(ctx) ? Optional.of(getWebDriver(ctx)) : Optional.<WebDriver>absent();
    }

    private PhantomJSDriver newPhantomJSDriver(File phantomjsBinary) {
        if (!phantomjsBinary.exists() || !phantomjsBinary.canExecute()) {
            throw new RuntimeException("phantomjsBinary: '" + phantomjsBinary + "' is missing or not executable...");
        }

        final DesiredCapabilities capabilities = new DesiredCapabilities();

        capabilities.setCapability(PHANTOMJS_EXECUTABLE_PATH_PROPERTY, phantomjsBinary.getAbsolutePath());

        PhantomJSDriver driver = new PhantomJSDriver(capabilities);
        driver.manage().timeouts().implicitlyWait(10, TimeUnit.SECONDS);

        return driver;
    }

    private boolean haveFluentWebDriver(ExecutionContext ctx) {
        return ctx.find(FLUENT_WEBDRIVER_KEY).isPresent() && ((ThreadLocal<FluentWebDriver>) ctx.get(FLUENT_WEBDRIVER_KEY)).get() != null;
    }
}
