package notclive.security.email;

import notclive.security.email.configuration.EmailSecurityProperties;
import notclive.security.email.exceptions.InvalidEmailAddressException;
import notclive.security.email.exceptions.UnknownSecretException;
import notclive.security.email.principals.EmailSecurityEmailKnownPrincipal;
import notclive.security.email.principals.EmailSecurityUserPrincipal;
import notclive.security.email.rememberMe.EmailSecurityRememberMeServices;
import org.hibernate.validator.internal.constraintvalidators.EmailValidator;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.web.authentication.AbstractAuthenticationProcessingFilter;
import org.springframework.security.web.util.RedirectUrlBuilder;
import org.springframework.stereotype.Component;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.UUID;

import static java.lang.Boolean.parseBoolean;

@Component
public class EmailSecurityFilter extends AbstractAuthenticationProcessingFilter {

    private final EmailValidator emailValidator = new EmailValidator();
    private final EmailSecurityProperties properties;
    private final EmailSecurityRequestMatcher requestMatcher;
    private final EmailSecurityTokenRepository tokenRepository;
    private final EmailSecurityEmailSender emailSender;
    private final EmailSecurityRememberMeServices rememberMeServices;

    @Autowired
    protected EmailSecurityFilter(EmailSecurityProperties properties,
                                  EmailSecurityRequestMatcher requestMatcher,
                                  EmailSecurityTokenRepository tokenRepository,
                                  EmailSecurityEmailSender emailSender,
                                  EmailSecurityRememberMeServices rememberMeServices) {
        super(requestMatcher);
        this.properties = properties;
        this.requestMatcher = requestMatcher;
        this.tokenRepository = tokenRepository;
        this.emailSender = emailSender;
        this.rememberMeServices = rememberMeServices;
    }

    @Autowired
    public void setAuthenticationSuccessHandler(EmailSecuritySuccessHandler successHandler) {
        super.setAuthenticationSuccessHandler(successHandler);
    }

    @Override
    @Autowired
    public void setAuthenticationManager(AuthenticationManager authenticationManager) {
        super.setAuthenticationManager(authenticationManager);
    }

    @Override
    public Authentication attemptAuthentication(HttpServletRequest request, HttpServletResponse response) {
        if (requestMatcher.isProvidingEmailAddress(request)) {
            return authenticateEmailAddress(request);
        } else if (requestMatcher.secretHasBeenProvidedInAnotherSession()) {
            return authenticateUser(request, response);
        } else if (requestMatcher.isProvidingSecret(request)) {
            return authenticateSecret(request);
        }
        return null;
    }

    private Authentication authenticateEmailAddress(HttpServletRequest request) {
        String emailAddress = getValidEmailAddress(request);
        boolean rememberMe = parseBoolean(request.getParameter("remember-me"));
        String secret = generateSecret(emailAddress);
        tokenRepository.store(new EmailSecurityToken(emailAddress, secret, false));
        emailSender.send(emailAddress, generateSecretUrl(request, secret));
        return new EmailSecurityEmailKnownPrincipal(emailAddress, secret, rememberMe);
    }

    private String getValidEmailAddress(HttpServletRequest request) {
        String emailAddress = request.getParameter("email");
        if (emailValidator.isValid(emailAddress, null)) {
            return emailAddress;
        }
        throw new InvalidEmailAddressException();
    }

    private String generateSecret(String emailAddress) {
        return properties.getPredictableSecrets()
                ? emailAddress
                : UUID.randomUUID().toString();
    }

    private String generateSecretUrl(HttpServletRequest request, String secret) {
        RedirectUrlBuilder urlBuilder = new RedirectUrlBuilder();
        urlBuilder.setScheme(request.getScheme());
        urlBuilder.setServerName(request.getServerName());
        urlBuilder.setPort(request.getServerPort());
        urlBuilder.setContextPath(request.getContextPath());
        urlBuilder.setPathInfo(properties.getPaths().getSecret());
        urlBuilder.setQuery("secret=" + secret);
        return urlBuilder.getUrl();
    }

    private Authentication authenticateUser(HttpServletRequest request, HttpServletResponse response) {
        String emailAddress = (String) SecurityContextHolder.getContext().getAuthentication().getPrincipal();
        EmailSecurityUserPrincipal authentication = new EmailSecurityUserPrincipal(emailAddress);
        rememberMeServices.loginSuccess(request, response, authentication);
        return authentication;
    }

    private Authentication authenticateSecret(HttpServletRequest request) {
        String secret = request.getParameter("secret");
        EmailSecurityToken token = tokenRepository.authenticate(secret);
        if (token == null) {
            throw new UnknownSecretException();
        }
        // Don't authenticate user in this session when providing secret. This request might be from
        // a different computer/browser/session than that which initiated authentication
        return null;
    }
}
