package nexcore.sprout.spring.boot.autoconfigure.session;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.session.SessionInformation;
import org.springframework.security.core.session.SessionRegistry;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.session.ExpiringSession;
import org.springframework.session.FindByIndexNameSessionRepository;
import org.springframework.session.Session;
import org.springframework.session.SessionRepository;

/**
 * 현재 로그인 하려는 유저정보를 이용하여 Spring Session Redis의 index에서 해당 Session을 검색하고 해당
 * Session을 RedisSessionInfomation 형태로 변환하여 중복 로그인, 세션 동시제어등 처리
 * 
 * @author 09281
 */
public class RedisSessionRegistry implements SessionRegistry{

	private static final Log logger = LogFactory.getLog(RedisSessionRegistry.class);
	private final SessionRepository<ExpiringSession> sessionRepository;
	
	private static final String SPRING_SECURITY_CONTEXT = "SPRING_SECURITY_CONTEXT";
	
	@Autowired
	private ApplicationContext applicationContext;
	
	@Autowired
	public RedisSessionRegistry(FindByIndexNameSessionRepository<ExpiringSession> sessionRepository) {
		this.sessionRepository = sessionRepository;
	}

	/**
	 * 사용자가 로그인 할 때 사용자의 principal을 이용해 index의 모든 Session을 검색
	 * 검색된 세션중 만기되지않은 세션들의 list를 return
	 * 
	 * @see org.springframework.security.core.session.SessionRegistry#getAllSessions(java.lang.Object, boolean)
	 */
	@Override
	public List<SessionInformation> getAllSessions(Object principal, boolean includeExpiredSessions) {

		Collection<ExpiringSession> indexValue =  ((FindByIndexNameSessionRepository<ExpiringSession>) sessionRepository).findByIndexNameAndIndexValue(
				FindByIndexNameSessionRepository.PRINCIPAL_NAME_INDEX_NAME, getUserId(principal)).values();

		List<SessionInformation> sessionlist = new ArrayList<SessionInformation>();
		for (ExpiringSession sessionValue : indexValue) {
			SessionInformation sessionInformation = new SessionInformation(getPrincipal(sessionValue), sessionValue.getId(), new Date(sessionValue.getLastAccessedTime()));
			if (includeExpiredSessions || !sessionInformation.isExpired()) {
				sessionlist.add(sessionInformation);
			}
		}
		return sessionlist;
	}
	
	/**
	 * sessionId를 이용해 SessionInformation 생성
	 * 
	 * @see org.springframework.security.core.session.SessionRegistry#getSessionInformation(java.lang.String)
	 */
	@Override
	public SessionInformation getSessionInformation(String sessionId) {
		ExpiringSession session = this.sessionRepository.getSession(sessionId);
		if (session != null) {
			return new SessionInformation(getPrincipal(session), session.getId(), new Date(session.getLastAccessedTime()));
		}
		return null;
	}

	/**
	 * session에서 PrincipalName return
	 * 
	 * @param session
	 * @return
	 */
	private static Object getPrincipal(Session session) {
		String principalName = session.getAttribute(FindByIndexNameSessionRepository.PRINCIPAL_NAME_INDEX_NAME);
		if (principalName != null) {
			return principalName;
		}
		SecurityContext securityContext = session.getAttribute(SPRING_SECURITY_CONTEXT);
		if (securityContext != null && securityContext.getAuthentication() != null) {
			return securityContext.getAuthentication().getName();
		}
		return "";
	}

	/**
	 * 로그인 하려는 사용자의 principal에서 UserId를 return
	 * 
	 * @param principal
	 * @return String
	 */
	private String getUserId(Object principal) {
		String userId = "";
		if (principal instanceof UserDetails) {
			userId = ((UserDetails) principal).getUsername();
		}
		return userId;
	}
	/**
	 * Spring Session에서 지원하지 않는 기능
	 */
	@Override
	public List<Object> getAllPrincipals() {
		throw new UnsupportedOperationException();
	}
	
	/**
	 * Spring Session에서 실행됨
	 */
	@Override
	public void refreshLastRequest(String sessionId) {
		ExpiringSession session = this.sessionRepository.getSession(sessionId);
		session.setLastAccessedTime(System.currentTimeMillis());
		sessionRepository.save(session);
	}

	/**
	 * Spring Session에서 실행됨
	 */
	@Override
	public void registerNewSession(String sessionId, Object principal) {
	}
	/**
	 * Redis 에서 sessionId에 해당하는 Session을 제거
	 */
	@Override
	public void removeSessionInformation(String sessionId) {
		this.sessionRepository.delete(sessionId);
	}
	
	
}
